/*
 * Decompiled with CFR 0.152.
 */
package org.sonar.scanner.http;

import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import javax.annotation.CheckForNull;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.sonar.api.notifications.AnalysisWarnings;
import org.sonar.api.utils.MessageException;
import org.sonar.api.utils.Preconditions;
import org.sonar.scanner.bootstrap.GlobalAnalysisMode;
import org.sonar.scanner.http.ScannerWsClient;
import org.sonarqube.ws.client.HttpException;
import org.sonarqube.ws.client.WsClient;
import org.sonarqube.ws.client.WsConnector;
import org.sonarqube.ws.client.WsRequest;
import org.sonarqube.ws.client.WsResponse;

public class DefaultScannerWsClient
implements ScannerWsClient {
    private static final int MAX_ERROR_MSG_LEN = 128;
    static final String SQ_TOKEN_EXPIRATION_HEADER = "SonarQube-Authentication-Token-Expiration";
    private static final DateTimeFormatter USER_FRIENDLY_DATETIME_FORMAT = DateTimeFormatter.ofPattern("MMMM dd, yyyy");
    private static final Logger LOG = LoggerFactory.getLogger(DefaultScannerWsClient.class);
    private final Set<String> warningMessages = new HashSet<String>();
    private final WsClient target;
    private final boolean hasCredentials;
    private final GlobalAnalysisMode globalMode;
    private final AnalysisWarnings analysisWarnings;

    public DefaultScannerWsClient(WsClient target, boolean hasCredentials, GlobalAnalysisMode globalMode, AnalysisWarnings analysisWarnings) {
        this.target = target;
        this.hasCredentials = hasCredentials;
        this.globalMode = globalMode;
        this.analysisWarnings = analysisWarnings;
    }

    @Override
    public WsResponse call(WsRequest request) {
        Preconditions.checkState(!this.globalMode.isMediumTest(), "No WS call should be made in medium test mode");
        WsResponse response = this.target.wsConnector().call(request);
        this.failIfUnauthorized(response);
        this.checkAuthenticationWarnings(response);
        return response;
    }

    @Override
    public String baseUrl() {
        return this.target.wsConnector().baseUrl();
    }

    WsConnector wsConnector() {
        return this.target.wsConnector();
    }

    private void failIfUnauthorized(WsResponse response) {
        String jsonMsg;
        int code = response.code();
        if (code == 401) {
            DefaultScannerWsClient.logResponseDetailsIfDebug(response);
            response.close();
            if (this.hasCredentials) {
                throw MessageException.of(String.format("Not authorized. Please check the user token in the property '%s' or '%s' (deprecated).", "sonar.token", "sonar.login"));
            }
            throw MessageException.of(String.format("Not authorized. Analyzing this project requires authentication. Please check the user token in the property '%s' or '%s' (deprecated).", "sonar.token", "sonar.login"));
        }
        if (code == 403) {
            DefaultScannerWsClient.logResponseDetailsIfDebug(response);
            throw MessageException.of("You're not authorized to analyze this project or the project doesn't exist on SonarQube and you're not authorized to create it. Please contact an administrator.");
        }
        if (code == 400 && (jsonMsg = DefaultScannerWsClient.tryParseAsJsonError(response.content())) != null) {
            throw MessageException.of(jsonMsg);
        }
        response.failIfNotSuccessful();
    }

    private static void logResponseDetailsIfDebug(WsResponse response) {
        if (LOG.isDebugEnabled()) {
            String content = response.hasContent() ? response.content() : "<no content>";
            Map<String, List<String>> headers = response.headers();
            LOG.debug("Error response content: {}, headers: {}", (Object)content, (Object)headers);
        }
    }

    private void checkAuthenticationWarnings(WsResponse response) {
        if (response.code() == 200) {
            response.header(SQ_TOKEN_EXPIRATION_HEADER).ifPresent(expirationDate -> {
                ZonedDateTime datetimeInUTC = ZonedDateTime.from(DateTimeFormatter.ofPattern("yyyy-MM-dd'T'HH:mm:ssZ").parse((CharSequence)expirationDate)).withZoneSameInstant(ZoneOffset.UTC);
                if (DefaultScannerWsClient.isTokenExpiringInOneWeek(datetimeInUTC)) {
                    this.addAnalysisWarning(datetimeInUTC);
                }
            });
        }
    }

    private static boolean isTokenExpiringInOneWeek(ZonedDateTime expirationDate) {
        ZonedDateTime localDateTime = ZonedDateTime.now(ZoneOffset.UTC);
        ZonedDateTime headerDateTime = expirationDate.minusDays(7L);
        return localDateTime.isAfter(headerDateTime);
    }

    private void addAnalysisWarning(ZonedDateTime tokenExpirationDate) {
        String warningMessage = "The token used for this analysis will expire on: " + tokenExpirationDate.format(USER_FRIENDLY_DATETIME_FORMAT);
        if (!this.warningMessages.contains(warningMessage)) {
            this.warningMessages.add(warningMessage);
            LOG.warn(warningMessage);
            LOG.warn("Analysis executed with this token will fail after the expiration date.");
        }
        this.analysisWarnings.addUnique(warningMessage + "\nAfter this date, the token can no longer be used to execute the analysis. Please consider generating a new token and updating it in the locations where it is in use.");
    }

    public static String createErrorMessage(HttpException exception) {
        String json = DefaultScannerWsClient.tryParseAsJsonError(exception.content());
        if (json != null) {
            return json;
        }
        String msg = "HTTP code " + exception.code();
        if (DefaultScannerWsClient.isHtml(exception.content())) {
            return msg;
        }
        return msg + ": " + StringUtils.left(exception.content(), 128);
    }

    @CheckForNull
    private static String tryParseAsJsonError(String responseContent) {
        try {
            JsonObject obj = JsonParser.parseString(responseContent).getAsJsonObject();
            JsonArray errors = obj.getAsJsonArray("errors");
            ArrayList<String> errorMessages = new ArrayList<String>();
            for (JsonElement e : errors) {
                errorMessages.add(e.getAsJsonObject().get("msg").getAsString());
            }
            return String.join((CharSequence)", ", errorMessages);
        }
        catch (Exception e) {
            return null;
        }
    }

    private static boolean isHtml(String responseContent) {
        return StringUtils.stripToEmpty(responseContent).startsWith("<!DOCTYPE html>");
    }
}

