/*
 * Decompiled with CFR 0.152.
 */
package org.apache.sshd.common.util.io;

import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.Closeable;
import java.io.EOFException;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.Reader;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.nio.file.AccessMode;
import java.nio.file.CopyOption;
import java.nio.file.FileSystem;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.PosixFilePermission;
import java.nio.file.attribute.UserPrincipal;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.EnumSet;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import org.apache.sshd.common.util.ExceptionUtils;
import org.apache.sshd.common.util.GenericUtils;
import org.apache.sshd.common.util.OsUtils;

public final class IoUtils {
    public static final OpenOption[] EMPTY_OPEN_OPTIONS = new OpenOption[0];
    public static final CopyOption[] EMPTY_COPY_OPTIONS = new CopyOption[0];
    public static final LinkOption[] EMPTY_LINK_OPTIONS = new LinkOption[0];
    public static final FileAttribute<?>[] EMPTY_FILE_ATTRIBUTES = new FileAttribute[0];
    public static final List<String> WINDOWS_EXECUTABLE_EXTENSIONS = Collections.unmodifiableList(Arrays.asList(".bat", ".exe", ".cmd"));
    public static final String REGFILE_VIEW_ATTR = "isRegularFile";
    public static final String DIRECTORY_VIEW_ATTR = "isDirectory";
    public static final String SYMLINK_VIEW_ATTR = "isSymbolicLink";
    public static final String NUMLINKS_VIEW_ATTR = "nlink";
    public static final String OTHERFILE_VIEW_ATTR = "isOther";
    public static final String EXECUTABLE_VIEW_ATTR = "isExecutable";
    public static final String SIZE_VIEW_ATTR = "size";
    public static final String OWNER_VIEW_ATTR = "owner";
    public static final String GROUP_VIEW_ATTR = "group";
    public static final String USERID_VIEW_ATTR = "uid";
    public static final String GROUPID_VIEW_ATTR = "gid";
    public static final String PERMISSIONS_VIEW_ATTR = "permissions";
    public static final String ACL_VIEW_ATTR = "acl";
    public static final String FILEKEY_VIEW_ATTR = "fileKey";
    public static final String CREATE_TIME_VIEW_ATTR = "creationTime";
    public static final String LASTMOD_TIME_VIEW_ATTR = "lastModifiedTime";
    public static final String LASTACC_TIME_VIEW_ATTR = "lastAccessTime";
    public static final String EXTENDED_VIEW_ATTR = "extended";
    public static final int DEFAULT_COPY_SIZE = 8192;
    public static final String EOL = System.lineSeparator();
    public static final Set<StandardOpenOption> WRITEABLE_OPEN_OPTIONS = Collections.unmodifiableSet(EnumSet.of(StandardOpenOption.APPEND, new StandardOpenOption[]{StandardOpenOption.CREATE, StandardOpenOption.CREATE_NEW, StandardOpenOption.DELETE_ON_CLOSE, StandardOpenOption.DSYNC, StandardOpenOption.SYNC, StandardOpenOption.TRUNCATE_EXISTING, StandardOpenOption.WRITE}));
    private static final byte[] EOL_BYTES = EOL.getBytes(StandardCharsets.UTF_8);
    private static final LinkOption[] NO_FOLLOW_OPTIONS = new LinkOption[]{LinkOption.NOFOLLOW_LINKS};

    private IoUtils() {
        throw new UnsupportedOperationException("No instance allowed");
    }

    public static byte[] getEOLBytes() {
        return (byte[])EOL_BYTES.clone();
    }

    public static LinkOption[] getLinkOptions(boolean followLinks) {
        if (followLinks) {
            return EMPTY_LINK_OPTIONS;
        }
        return (LinkOption[])NO_FOLLOW_OPTIONS.clone();
    }

    public static long copy(InputStream source2, OutputStream sink2) throws IOException {
        return IoUtils.copy(source2, sink2, 8192);
    }

    public static long copy(InputStream source2, OutputStream sink2, int bufferSize) throws IOException {
        long nread = 0L;
        byte[] buf = new byte[bufferSize];
        int n = source2.read(buf);
        while (n > 0) {
            sink2.write(buf, 0, n);
            nread += (long)n;
            n = source2.read(buf);
        }
        return nread;
    }

    public static IOException closeQuietly(Closeable ... closeables) {
        return IoUtils.closeQuietly(GenericUtils.isEmpty(closeables) ? Collections.emptyList() : Arrays.asList(closeables));
    }

    public static IOException closeQuietly(Closeable c) {
        if (c != null) {
            try {
                c.close();
            }
            catch (IOException e) {
                return e;
            }
        }
        return null;
    }

    public static IOException closeQuietly(Collection<? extends Closeable> closeables) {
        if (GenericUtils.isEmpty(closeables)) {
            return null;
        }
        IOException err = null;
        for (Closeable closeable : closeables) {
            try {
                if (closeable == null) continue;
                closeable.close();
            }
            catch (IOException e) {
                err = ExceptionUtils.accumulateException(err, e);
            }
        }
        return err;
    }

    public static boolean isWindowsExecutable(String fileName) {
        if (fileName == null || fileName.length() <= 0) {
            return false;
        }
        for (String suffix : WINDOWS_EXECUTABLE_EXTENSIONS) {
            if (!fileName.endsWith(suffix)) continue;
            return true;
        }
        return false;
    }

    public static Set<PosixFilePermission> getPermissions(Path path, LinkOption ... options) throws IOException {
        FileSystem fs = path.getFileSystem();
        Set<String> views = fs.supportedFileAttributeViews();
        if (views.contains("posix")) {
            return Files.getPosixFilePermissions(path, options);
        }
        return IoUtils.getPermissionsFromFile(path.toFile());
    }

    public static Set<PosixFilePermission> getPermissionsFromFile(File f) {
        EnumSet<PosixFilePermission> perms = EnumSet.noneOf(PosixFilePermission.class);
        if (f.canRead()) {
            perms.add(PosixFilePermission.OWNER_READ);
            perms.add(PosixFilePermission.GROUP_READ);
            perms.add(PosixFilePermission.OTHERS_READ);
        }
        if (f.canWrite()) {
            perms.add(PosixFilePermission.OWNER_WRITE);
            perms.add(PosixFilePermission.GROUP_WRITE);
            perms.add(PosixFilePermission.OTHERS_WRITE);
        }
        if (IoUtils.isExecutable(f)) {
            perms.add(PosixFilePermission.OWNER_EXECUTE);
            perms.add(PosixFilePermission.GROUP_EXECUTE);
            perms.add(PosixFilePermission.OTHERS_EXECUTE);
        }
        return perms;
    }

    public static boolean isExecutable(File f) {
        if (f == null) {
            return false;
        }
        if (OsUtils.isWin32()) {
            return IoUtils.isWindowsExecutable(f.getName());
        }
        return f.canExecute();
    }

    public static void setPermissions(Path path, Set<PosixFilePermission> perms) throws IOException {
        FileSystem fs = path.getFileSystem();
        Set<String> views = fs.supportedFileAttributeViews();
        if (views.contains("posix")) {
            Files.setPosixFilePermissions(path, perms);
        } else {
            IoUtils.setPermissionsToFile(path.toFile(), perms);
        }
    }

    public static void setPermissionsToFile(File f, Collection<PosixFilePermission> perms) {
        boolean havePermissions = GenericUtils.isNotEmpty(perms);
        boolean readable = havePermissions && (perms.contains((Object)PosixFilePermission.OWNER_READ) || perms.contains((Object)PosixFilePermission.GROUP_READ) || perms.contains((Object)PosixFilePermission.OTHERS_READ));
        f.setReadable(readable, false);
        boolean writable = havePermissions && (perms.contains((Object)PosixFilePermission.OWNER_WRITE) || perms.contains((Object)PosixFilePermission.GROUP_WRITE) || perms.contains((Object)PosixFilePermission.OTHERS_WRITE));
        f.setWritable(writable, false);
        boolean executable = havePermissions && (perms.contains((Object)PosixFilePermission.OWNER_EXECUTE) || perms.contains((Object)PosixFilePermission.GROUP_EXECUTE) || perms.contains((Object)PosixFilePermission.OTHERS_EXECUTE));
        f.setExecutable(executable, false);
    }

    public static String getFileOwner(Path path, LinkOption ... options) throws IOException {
        try {
            UserPrincipal principal = Files.getOwner(path, options);
            String owner = principal == null ? null : principal.getName();
            return OsUtils.getCanonicalUser(owner);
        }
        catch (UnsupportedOperationException e) {
            return null;
        }
    }

    public static Boolean checkFileExists(Path path, LinkOption ... options) {
        boolean followLinks = IoUtils.followLinks(options);
        try {
            if (followLinks) {
                path.getFileSystem().provider().checkAccess(path, new AccessMode[0]);
            } else {
                Files.readAttributes(path, BasicFileAttributes.class, options);
            }
            return Boolean.TRUE;
        }
        catch (NoSuchFileException e) {
            return Boolean.FALSE;
        }
        catch (IOException e) {
            return null;
        }
    }

    public static Boolean checkFileExistsAnySymlinks(Path path, boolean neverFollowSymlinks) {
        try {
            if (!neverFollowSymlinks) {
                path.getFileSystem().provider().checkAccess(path, new AccessMode[0]);
            } else {
                for (int i2 = 1; i2 <= path.getNameCount(); ++i2) {
                    Path checkForSymLink = IoUtils.getFirstPartsOfPath(path, i2);
                    BasicFileAttributes basicFileAttributes = Files.readAttributes(checkForSymLink, BasicFileAttributes.class, LinkOption.NOFOLLOW_LINKS);
                    if (!basicFileAttributes.isSymbolicLink()) continue;
                    return false;
                }
            }
            return true;
        }
        catch (NoSuchFileException e) {
            return false;
        }
        catch (IOException e) {
            return null;
        }
    }

    public static Path getFirstPartsOfPath(Path path, int partsToExtract) {
        String firstName = path.getName(0).toString();
        String[] names = new String[partsToExtract - 1];
        for (int j = 1; j < partsToExtract; ++j) {
            names[j - 1] = path.getName(j).toString();
        }
        Path checkForSymLink = path.getFileSystem().getPath(firstName, names);
        Path root = path.getRoot();
        if (root != null) {
            checkForSymLink = root.resolve(checkForSymLink);
        }
        return checkForSymLink;
    }

    public static void readFully(InputStream input, byte[] buffer) throws IOException {
        IoUtils.readFully(input, buffer, 0, buffer.length);
    }

    public static void readFully(InputStream input, byte[] buffer, int offset, int length) throws IOException {
        int actual = IoUtils.read(input, buffer, offset, length);
        if (actual != length) {
            throw new EOFException("Premature EOF - expected=" + length + ", actual=" + actual);
        }
    }

    public static int read(InputStream input, byte[] buffer) throws IOException {
        return IoUtils.read(input, buffer, 0, buffer.length);
    }

    public static int read(InputStream input, byte[] buffer, int offset, int length) throws IOException {
        int remaining = length;
        int curOffset = offset;
        while (remaining > 0) {
            int count = input.read(buffer, curOffset, remaining);
            if (count == -1) {
                return curOffset - offset;
            }
            remaining -= count;
            curOffset += count;
        }
        return length;
    }

    public static PosixFilePermission validateExcludedPermissions(Collection<PosixFilePermission> perms, Collection<PosixFilePermission> excluded) {
        if (GenericUtils.isEmpty(perms) || GenericUtils.isEmpty(excluded)) {
            return null;
        }
        for (PosixFilePermission p : excluded) {
            if (!perms.contains((Object)p)) continue;
            return p;
        }
        return null;
    }

    public static Path ensureDirectory(Path path, LinkOption ... options) {
        if (!Files.isDirectory(path, options)) {
            throw new UnsupportedOperationException("Not a directory: " + path);
        }
        return path;
    }

    public static boolean followLinks(LinkOption ... options) {
        if (GenericUtils.isEmpty(options)) {
            return true;
        }
        for (LinkOption localLinkOption : options) {
            if (localLinkOption != LinkOption.NOFOLLOW_LINKS) continue;
            return false;
        }
        return true;
    }

    public static String appendPathComponent(String prefix, String component) {
        if (GenericUtils.isEmpty(prefix)) {
            return component;
        }
        if (GenericUtils.isEmpty(component)) {
            return prefix;
        }
        StringBuilder sb = new StringBuilder(prefix.length() + component.length() + File.separator.length()).append(prefix);
        if (sb.charAt(prefix.length() - 1) == File.separatorChar) {
            if (component.charAt(0) == File.separatorChar) {
                sb.append(component.substring(1));
            } else {
                sb.append(component);
            }
        } else {
            if (component.charAt(0) != File.separatorChar) {
                sb.append(File.separatorChar);
            }
            sb.append(component);
        }
        return sb.toString();
    }

    public static byte[] toByteArray(InputStream inStream) throws IOException {
        try (ByteArrayOutputStream baos = new ByteArrayOutputStream(8192);){
            IoUtils.copy(inStream, baos);
            byte[] byArray = baos.toByteArray();
            return byArray;
        }
    }

    public static List<String> readAllLines(URL url) throws IOException {
        try (InputStream stream = Objects.requireNonNull(url, "No URL").openStream();){
            List<String> list = IoUtils.readAllLines(stream);
            return list;
        }
    }

    public static List<String> readAllLines(InputStream stream) throws IOException {
        try (InputStreamReader reader = new InputStreamReader(Objects.requireNonNull(stream, "No stream instance"), StandardCharsets.UTF_8);){
            List<String> list = IoUtils.readAllLines(reader);
            return list;
        }
    }

    public static List<String> readAllLines(Reader reader) throws IOException {
        try (BufferedReader br = new BufferedReader(Objects.requireNonNull(reader, "No reader instance"), 8192);){
            List<String> list = IoUtils.readAllLines(br);
            return list;
        }
    }

    public static List<String> readAllLines(BufferedReader reader) throws IOException {
        return IoUtils.readAllLines(reader, -1);
    }

    public static List<String> readAllLines(BufferedReader reader, int lineCountHint) throws IOException {
        ArrayList<String> result = new ArrayList<String>(Math.max(lineCountHint, 16));
        String line = reader.readLine();
        while (line != null) {
            result.add(line);
            line = reader.readLine();
        }
        return result;
    }

    public static Path chroot(Path newRoot, Path toSanitize) {
        Objects.requireNonNull(newRoot);
        Objects.requireNonNull(toSanitize);
        List<String> sanitized = IoUtils.removeExtraCdUps(toSanitize);
        return IoUtils.buildPath(newRoot, newRoot.getFileSystem(), sanitized);
    }

    public static Path removeCdUpAboveRoot(Path toSanitize) {
        List<String> sanitized = IoUtils.removeExtraCdUps(toSanitize);
        return IoUtils.buildPath(toSanitize.getRoot(), toSanitize.getFileSystem(), sanitized);
    }

    private static List<String> removeExtraCdUps(Path toResolve) {
        ArrayList<String> newNames = new ArrayList<String>(toResolve.getNameCount());
        int numCdUps = 0;
        int numDirParts = 0;
        for (int i2 = 0; i2 < toResolve.getNameCount(); ++i2) {
            String name = toResolve.getName(i2).toString();
            if ("..".equals(name)) {
                if (numDirParts <= numCdUps) continue;
                ++numCdUps;
                newNames.add(name);
                continue;
            }
            if (!".".equals(name)) {
                ++numDirParts;
            }
            newNames.add(name);
        }
        return newNames;
    }

    public static Path buildPath(Path root, FileSystem fs, List<String> namesList) {
        Objects.requireNonNull(fs);
        if (namesList == null) {
            return null;
        }
        if (GenericUtils.isEmpty(namesList)) {
            return root == null ? fs.getPath(".", new String[0]) : root;
        }
        Path cleanedPathToResolve = IoUtils.buildRelativePath(fs, namesList);
        return root == null ? cleanedPathToResolve : root.resolve(cleanedPathToResolve);
    }

    public static Path buildRelativePath(FileSystem fs, List<String> namesList) {
        String[] names = new String[namesList.size() - 1];
        Iterator<String> it = namesList.iterator();
        String rootName = it.next();
        int i2 = 0;
        while (it.hasNext()) {
            names[i2] = it.next();
            ++i2;
        }
        Path cleanedPathToResolve = fs.getPath(rootName, names);
        return cleanedPathToResolve;
    }
}

