/*
 * Decompiled with CFR 0.152.
 */
package com.tplink.smb.ecsp.transport.netty;

import com.tplink.smb.ecsp.common.URL;
import com.tplink.smb.ecsp.common.util.encrypt.RC4Utils;
import com.tplink.smb.ecsp.protocol.packet.EcspPacket;
import com.tplink.smb.ecsp.transport.api.monitor.ServerMonitor;
import com.tplink.smb.ecsp.transport.netty.NettyChannel;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import io.netty.channel.ChannelHandler;
import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.socket.DatagramPacket;
import io.netty.handler.codec.ByteToMessageDecoder;
import io.netty.handler.codec.MessageToMessageDecoder;
import io.netty.handler.codec.MessageToMessageEncoder;
import java.net.InetSocketAddress;
import java.nio.ByteBuffer;
import java.util.List;
import java.util.Objects;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class NettyCodecAdapter {
    private static final Logger log = LoggerFactory.getLogger(NettyCodecAdapter.class);
    private final URL url;
    private final com.tplink.smb.ecsp.transport.api.ChannelHandler handler;

    NettyCodecAdapter(URL url, com.tplink.smb.ecsp.transport.api.ChannelHandler handler) {
        this.url = url;
        this.handler = handler;
    }

    ChannelHandler getDefaultEncoder() {
        return new TcpInternalEncoder();
    }

    ChannelHandler getDefaultDecoder() {
        return new TcpInternalDecoder();
    }

    ChannelHandler getUdpEncoder() {
        return new UdpInternalEncoder();
    }

    ChannelHandler getUdpDecoder() {
        return new UdpInternalDecoder();
    }

    ChannelHandler getEncryptEncoder(byte[] key) {
        return new TcpEncryptInternalEncoder(key);
    }

    ChannelHandler getEncryptDecoder(byte[] key) {
        return new TcpEncryptInternalDecoder(0xA00000, 4, 4, key);
    }

    ChannelHandler getFileEncoder() {
        return new ByteMessageEncoder();
    }

    ChannelHandler getFileDecoder() {
        return new ByteMessageDecoder();
    }

    private class TcpInternalEncoder
    extends MessageToMessageEncoder<EcspPacket> {
        private TcpInternalEncoder() {
        }

        protected void encode(ChannelHandlerContext ctx, EcspPacket msg, List<Object> out) {
            if (Objects.isNull(msg)) {
                throw new IllegalArgumentException("TCP Encoder error: msg is null");
            }
            NettyChannel.getOrAddChannel(ctx.channel(), NettyCodecAdapter.this.url, NettyCodecAdapter.this.handler);
            ByteBuf buf = ctx.alloc().buffer();
            buf.writeBytes(msg.getPayload());
            out.add(buf);
        }
    }

    private class TcpInternalDecoder
    extends ByteToMessageDecoder {
        private TcpInternalDecoder() {
        }

        protected void decode(ChannelHandlerContext ctx, ByteBuf input, List<Object> out) {
            NettyChannel.getOrAddChannel(ctx.channel(), NettyCodecAdapter.this.url, NettyCodecAdapter.this.handler);
            int ecspPacketLength = 4;
            int payloadLength = input.readableBytes();
            if (payloadLength < 4) {
                log.warn("Buf readable bytes is too little {}, wait for next tcp frame", (Object)payloadLength);
                return;
            }
            byte[] payloadBytes = new byte[payloadLength];
            input.readBytes(payloadBytes);
            EcspPacket ecspPacket = EcspPacket.builder().lengthField(payloadLength).payload(ByteBuffer.wrap(payloadBytes)).build();
            out.add(ecspPacket);
        }
    }

    private class UdpInternalEncoder
    extends MessageToMessageEncoder<EcspPacket> {
        private UdpInternalEncoder() {
        }

        protected void encode(ChannelHandlerContext ctx, EcspPacket msg, List<Object> out) {
            if (Objects.isNull(msg)) {
                throw new IllegalArgumentException("Udp Encoder error: msg is null");
            }
            NettyChannel.getOrAddChannel(ctx.channel(), NettyCodecAdapter.this.url, NettyCodecAdapter.this.handler);
            ByteBuf byteBuf = ctx.alloc().buffer();
            byteBuf.writeInt(msg.getLengthField());
            byteBuf.writeBytes(msg.getPayload());
            out.add(new DatagramPacket(byteBuf, msg.getRemoteAddress()));
        }
    }

    private class UdpInternalDecoder
    extends MessageToMessageDecoder<DatagramPacket> {
        private UdpInternalDecoder() {
        }

        protected void decode(ChannelHandlerContext ctx, DatagramPacket msg, List<Object> out) {
            if (Objects.isNull(msg)) {
                throw new IllegalArgumentException("Udp Decoder error: msg is null");
            }
            NettyChannel.getOrAddChannel(ctx.channel(), NettyCodecAdapter.this.url, NettyCodecAdapter.this.handler);
            ByteBuf buf = (ByteBuf)msg.content();
            if (buf.readableBytes() < 4) {
                log.warn("UDP buf readable bytes is too little {}", (Object)buf.readableBytes());
                ServerMonitor.incUdpDecodeFailedCounter();
                return;
            }
            int payloadLength = buf.readInt();
            if (payloadLength > 2000) {
                log.warn("The length of the UDP package is {}, exceeds the maximum length {}", (Object)payloadLength, (Object)2000);
                ServerMonitor.incUdpDecodeFailedCounter();
                return;
            }
            if (payloadLength != buf.readableBytes()) {
                log.warn("The length of the UDP package is {}, not equals real length {}", (Object)payloadLength, (Object)buf.readableBytes());
                ServerMonitor.incUdpDecodeFailedCounter();
                return;
            }
            EcspPacket ecspPacket = EcspPacket.builder().lengthField(payloadLength).payload(buf.nioBuffer()).localAddress((InetSocketAddress)msg.recipient()).remoteAddress((InetSocketAddress)msg.sender()).build();
            out.add(ecspPacket);
        }
    }

    private class TcpEncryptInternalEncoder
    extends MessageToMessageEncoder<ByteBuf> {
        private final byte[] encryptKey;

        TcpEncryptInternalEncoder(byte[] encryptKey) {
            if (encryptKey == null) {
                throw new NullPointerException("encryptKey");
            }
            this.encryptKey = (byte[])encryptKey.clone();
        }

        protected void encode(ChannelHandlerContext ctx, ByteBuf msg, List<Object> out) {
            if (msg == null) {
                throw new IllegalArgumentException("TCP Encrypt Encoder error: msg is null");
            }
            NettyChannel.getOrAddChannel(ctx.channel(), NettyCodecAdapter.this.url, NettyCodecAdapter.this.handler);
            int length = msg.readableBytes();
            byte[] lengthBytes = this.intToByteArray(length);
            byte[] encryptedLength = RC4Utils.crypt((byte[])lengthBytes, (byte[])this.encryptKey);
            out.add(ctx.alloc().buffer(4).writeBytes(encryptedLength));
            byte[] msgBytes = new byte[length];
            msg.readBytes(msgBytes);
            byte[] encryptedMsg = RC4Utils.crypt((byte[])msgBytes, (byte[])this.encryptKey);
            out.add(ctx.alloc().buffer().writeBytes(encryptedMsg));
        }

        private byte[] intToByteArray(int value) {
            return new byte[]{(byte)(value >> 24), (byte)(value >> 16), (byte)(value >> 8), (byte)value};
        }
    }

    private class TcpEncryptInternalDecoder
    extends ByteToMessageDecoder {
        private final int lengthFieldLength;
        private final int maxFrameLength;
        private final int initialBytesToStrip;
        private final boolean failFast;
        private boolean discardingTooLongFrame;
        private long tooLongFrameLength;
        private long bytesToDiscard;
        private final byte[] decryptKey;

        TcpEncryptInternalDecoder(int maxFrameLength, int lengthFieldLength, int initialBytesToStrip, byte[] decryptKey) {
            this(maxFrameLength, lengthFieldLength, initialBytesToStrip, decryptKey, true);
        }

        TcpEncryptInternalDecoder(int maxFrameLength, int lengthFieldLength, int initialBytesToStrip, byte[] decryptKey, boolean failFast) {
            if (maxFrameLength <= 0) {
                throw new IllegalArgumentException("maxFrameLength must be a positive integer: " + maxFrameLength);
            }
            if (initialBytesToStrip < 0) {
                throw new IllegalArgumentException("initialBytesToStrip must be a non-negative integer: " + initialBytesToStrip);
            }
            if (decryptKey == null) {
                throw new NullPointerException("decryptKey");
            }
            this.maxFrameLength = maxFrameLength;
            this.lengthFieldLength = lengthFieldLength;
            this.initialBytesToStrip = initialBytesToStrip;
            this.failFast = failFast;
            this.decryptKey = (byte[])decryptKey.clone();
        }

        protected final void decode(ChannelHandlerContext ctx, ByteBuf in, List<Object> out) {
            NettyChannel.getOrAddChannel(ctx.channel(), NettyCodecAdapter.this.url, NettyCodecAdapter.this.handler);
            Object decoded = this.decode(ctx, in);
            if (decoded != null) {
                out.add(decoded);
            }
        }

        Object decode(ChannelHandlerContext ctx, ByteBuf in) {
            if (this.discardingTooLongFrame) {
                long bytesToDiscardTemp = this.bytesToDiscard;
                int localBytesToDiscard = (int)Math.min(bytesToDiscardTemp, (long)in.readableBytes());
                in.skipBytes(localBytesToDiscard);
                this.bytesToDiscard = bytesToDiscardTemp -= (long)localBytesToDiscard;
                this.failIfNecessary(ctx, false);
            }
            if (in.readableBytes() < this.lengthFieldLength) {
                log.debug("Buf readable bytes {} is too little, wait for next encrypt frame", (Object)in.readableBytes());
                return null;
            }
            long frameLength = this.getUnadjustedFrameLength(in, this.lengthFieldLength);
            if (frameLength < 0L) {
                in.skipBytes(this.lengthFieldLength);
                ServerMonitor.incRc4DecodeFailedCounter();
                log.warn("negative pre-adjustment length field: {}", (Object)frameLength);
                ctx.channel().close();
                return null;
            }
            if ((frameLength += (long)this.lengthFieldLength) > (long)this.maxFrameLength) {
                long discard = frameLength - (long)in.readableBytes();
                this.tooLongFrameLength = frameLength;
                if (discard >= 0L) {
                    this.discardingTooLongFrame = true;
                    this.bytesToDiscard = discard;
                    in.skipBytes(in.readableBytes());
                } else {
                    in.skipBytes((int)frameLength);
                }
                this.failIfNecessary(ctx, true);
                return null;
            }
            int frameLengthInt = (int)frameLength;
            if (in.readableBytes() < frameLengthInt) {
                return null;
            }
            if (this.initialBytesToStrip > frameLengthInt) {
                in.skipBytes(frameLengthInt);
                ServerMonitor.incRc4DecodeFailedCounter();
                log.warn("Adjusted frame length ({}) is less than initialBytesToStrip: {}", (Object)frameLength, (Object)this.initialBytesToStrip);
                ctx.channel().close();
                return null;
            }
            in.skipBytes(this.initialBytesToStrip);
            int readerIndex = in.readerIndex();
            int actualFrameLength = frameLengthInt - this.initialBytesToStrip;
            ByteBuf frame = this.extractFrame(ctx, in, readerIndex, actualFrameLength);
            in.readerIndex(readerIndex + actualFrameLength);
            return frame;
        }

        long getUnadjustedFrameLength(ByteBuf buf, int length) {
            long frameLength;
            if (length == 4) {
                byte[] lengthBytes = new byte[4];
                buf.readBytes(lengthBytes);
                buf.readerIndex(buf.readerIndex() - 4);
                byte[] decryptedLengthBytes = RC4Utils.crypt((byte[])lengthBytes, (byte[])this.decryptKey);
                frameLength = this.byteArrayToInt(decryptedLengthBytes);
            } else {
                log.info("unsupported lengthFieldLength: {}, expected: 4", (Object)this.lengthFieldLength);
                frameLength = -1L;
            }
            return frameLength;
        }

        private void failIfNecessary(ChannelHandlerContext ctx, boolean firstDetectionOfTooLongFrame) {
            if (this.bytesToDiscard == 0L) {
                long tooLongFrameLengthTemp = this.tooLongFrameLength;
                this.tooLongFrameLength = 0L;
                this.discardingTooLongFrame = false;
                if (!this.failFast || firstDetectionOfTooLongFrame) {
                    this.fail(ctx, tooLongFrameLengthTemp);
                }
            } else if (this.failFast && firstDetectionOfTooLongFrame) {
                this.fail(ctx, this.tooLongFrameLength);
            }
        }

        ByteBuf extractFrame(ChannelHandlerContext ctx, ByteBuf buffer, int index, int length) {
            byte[] contentBytes = new byte[length];
            buffer.slice(index, length).readBytes(contentBytes);
            byte[] contentBytesDecrypted = RC4Utils.crypt((byte[])contentBytes, (byte[])this.decryptKey);
            ByteBuf frame = ctx.alloc().buffer(length);
            frame.writeBytes(contentBytesDecrypted);
            return frame;
        }

        private void fail(ChannelHandlerContext ctx, long frameLength) {
            ServerMonitor.incRc4DecodeFailedCounter();
            if (frameLength > 0L) {
                log.warn("Adjusted frame length exceeds {} : {} - discarded", (Object)this.maxFrameLength, (Object)frameLength);
            } else {
                log.warn("Adjusted frame length exceeds {} - discarded", (Object)this.maxFrameLength);
            }
            ctx.channel().close();
        }

        private int byteArrayToInt(byte[] bytes) {
            if (bytes.length != 4) {
                return 0;
            }
            return (bytes[0] & 0xFF) << 24 | (bytes[1] & 0xFF) << 16 | (bytes[2] & 0xFF) << 8 | bytes[3] & 0xFF;
        }
    }

    private static class ByteMessageEncoder
    extends MessageToMessageEncoder<byte[]> {
        private ByteMessageEncoder() {
        }

        protected void encode(ChannelHandlerContext ctx, byte[] msg, List<Object> out) {
            out.add(Unpooled.wrappedBuffer((byte[])msg));
        }
    }

    private static class ByteMessageDecoder
    extends ByteToMessageDecoder {
        private ByteMessageDecoder() {
        }

        protected void decode(ChannelHandlerContext ctx, ByteBuf input, List<Object> out) {
            int payloadLength = input.readableBytes();
            byte[] payloadBytes = new byte[payloadLength];
            input.readBytes(payloadBytes);
            out.add(payloadBytes);
        }
    }
}

