// src/options.js
var EsVersion = {
  ES2018: 2018,
  ES2024: 2024,
  ES2025: 2025
};
function getOptions(options) {
  if (options?.target !== void 0 && !EsVersion[options.target]) {
    throw new Error(`Unexpected target "${options.target}"`);
  }
  return {
    // Sets the level of emulation rigor/strictness.
    accuracy: "default",
    // Disables advanced emulation strategies that rely on returning a `RegExp` subclass, resulting
    // in certain patterns not being emulatable.
    avoidSubclass: false,
    // Oniguruma flags; a string with `i`, `m`, and `x` in any order (all optional). Oniguruma's
    // `m` is equivalent to JavaScript's `s` (`dotAll`).
    flags: "",
    // Include JavaScript flag `g` (`global`) in the result.
    global: false,
    // Include JavaScript flag `d` (`hasIndices`) in the result.
    hasIndices: false,
    // Specifies the recursion depth limit. Supported values are integers `2` to `100` and `null`.
    // If `null`, any use of recursion results in an error.
    maxRecursionDepth: 6,
    // Sets the JavaScript language version for the generated pattern and flags. Later targets
    // allow faster processing, simpler generated source, and support for additional features.
    target: "ES2024",
    // Leave disabled unless the regex will be used in a TextMate grammar processor that merges
    // backreferences across `begin` and `end` patterns.
    tmGrammar: false,
    // Disables optimizations that simplify the pattern when it doesn't change the meaning.
    verbose: false,
    ...options
  };
}

// src/utils.js
var cp = String.fromCodePoint;
var r2 = String.raw;
function getNewCurrentFlags(current, { enable, disable }) {
  return {
    dotAll: !disable?.dotAll && !!(enable?.dotAll || current.dotAll),
    ignoreCase: !disable?.ignoreCase && !!(enable?.ignoreCase || current.ignoreCase)
  };
}
function getOrCreate(map, key, defaultValue) {
  if (!map.has(key)) {
    map.set(key, defaultValue);
  }
  return map.get(key);
}
function isMinTarget(target, min) {
  return EsVersion[target] >= EsVersion[min];
}
function throwIfNot(value, msg) {
  if (!value) {
    throw new Error(msg ?? "Value expected");
  }
  return value;
}

// src/unicode.js
var CharsWithoutIgnoreCaseExpansion = /* @__PURE__ */ new Set([
  cp(304),
  // İ
  cp(305)
  // ı
]);
function getIgnoreCaseMatchChars(char) {
  if (CharsWithoutIgnoreCaseExpansion.has(char)) {
    return [char];
  }
  const set = /* @__PURE__ */ new Set();
  const lower = char.toLowerCase();
  const upper = lower.toUpperCase();
  const title = LowerToTitleCaseMap.get(lower);
  const special = LowerToAlternativeUpperCaseMap.get(lower);
  if ([...upper].length === 1) {
    set.add(upper);
  }
  set.add(lower);
  title && set.add(title);
  special && set.add(special);
  return [...set];
}
var JsUnicodeProperties = /* @__PURE__ */ new Set([
  // ES2024 general categories and their aliases; all are supported by Oniguruma
  // See <github.com/mathiasbynens/unicode-match-property-value-ecmascript/blob/main/data/mappings.js>
  "C",
  "Other",
  "Cc",
  "Control",
  "cntrl",
  "Cf",
  "Format",
  "Cn",
  "Unassigned",
  "Co",
  "Private_Use",
  "Cs",
  "Surrogate",
  "L",
  "Letter",
  "LC",
  "Cased_Letter",
  "Ll",
  "Lowercase_Letter",
  "Lm",
  "Modifier_Letter",
  "Lo",
  "Other_Letter",
  "Lt",
  "Titlecase_Letter",
  "Lu",
  "Uppercase_Letter",
  "M",
  "Mark",
  "Combining_Mark",
  "Mc",
  "Spacing_Mark",
  "Me",
  "Enclosing_Mark",
  "Mn",
  "Nonspacing_Mark",
  "N",
  "Number",
  "Nd",
  "Decimal_Number",
  "digit",
  "Nl",
  "Letter_Number",
  "No",
  "Other_Number",
  "P",
  "Punctuation",
  "punct",
  "Pc",
  "Connector_Punctuation",
  "Pd",
  "Dash_Punctuation",
  "Pe",
  "Close_Punctuation",
  "Pf",
  "Final_Punctuation",
  "Pi",
  "Initial_Punctuation",
  "Po",
  "Other_Punctuation",
  "Ps",
  "Open_Punctuation",
  "S",
  "Symbol",
  "Sc",
  "Currency_Symbol",
  "Sk",
  "Modifier_Symbol",
  "Sm",
  "Math_Symbol",
  "So",
  "Other_Symbol",
  "Z",
  "Separator",
  "Zl",
  "Line_Separator",
  "Zp",
  "Paragraph_Separator",
  "Zs",
  "Space_Separator",
  // ES2024 binary properties and their aliases; all are supported by Oniguruma
  // See <tc39.es/ecma262/multipage/text-processing.html#table-binary-unicode-properties>
  "ASCII",
  "ASCII_Hex_Digit",
  "AHex",
  "Alphabetic",
  "Alpha",
  "Any",
  "Assigned",
  "Bidi_Control",
  "Bidi_C",
  "Bidi_Mirrored",
  "Bidi_M",
  "Case_Ignorable",
  "CI",
  "Cased",
  "Changes_When_Casefolded",
  "CWCF",
  "Changes_When_Casemapped",
  "CWCM",
  "Changes_When_Lowercased",
  "CWL",
  "Changes_When_NFKC_Casefolded",
  "CWKCF",
  "Changes_When_Titlecased",
  "CWT",
  "Changes_When_Uppercased",
  "CWU",
  "Dash",
  "Default_Ignorable_Code_Point",
  "DI",
  "Deprecated",
  "Dep",
  "Diacritic",
  "Dia",
  "Emoji",
  "Emoji_Component",
  "EComp",
  "Emoji_Modifier",
  "EMod",
  "Emoji_Modifier_Base",
  "EBase",
  "Emoji_Presentation",
  "EPres",
  "Extended_Pictographic",
  "ExtPict",
  "Extender",
  "Ext",
  "Grapheme_Base",
  "Gr_Base",
  "Grapheme_Extend",
  "Gr_Ext",
  "Hex_Digit",
  "Hex",
  "IDS_Binary_Operator",
  "IDSB",
  "IDS_Trinary_Operator",
  "IDST",
  "ID_Continue",
  "IDC",
  "ID_Start",
  "IDS",
  "Ideographic",
  "Ideo",
  "Join_Control",
  "Join_C",
  "Logical_Order_Exception",
  "LOE",
  "Lowercase",
  "Lower",
  "Math",
  "Noncharacter_Code_Point",
  "NChar",
  "Pattern_Syntax",
  "Pat_Syn",
  "Pattern_White_Space",
  "Pat_WS",
  "Quotation_Mark",
  "QMark",
  "Radical",
  "Regional_Indicator",
  "RI",
  "Sentence_Terminal",
  "STerm",
  "Soft_Dotted",
  "SD",
  "Terminal_Punctuation",
  "Term",
  "Unified_Ideograph",
  "UIdeo",
  "Uppercase",
  "Upper",
  "Variation_Selector",
  "VS",
  "White_Space",
  "space",
  "XID_Continue",
  "XIDC",
  "XID_Start",
  "XIDS"
]);
var JsUnicodePropertiesMap = /* @__PURE__ */ new Map();
for (const p of JsUnicodeProperties) {
  JsUnicodePropertiesMap.set(slug(p), p);
}
var JsUnicodePropertiesOfStrings = /* @__PURE__ */ new Set([
  // ES2024 properties of strings; none are supported by Oniguruma
  "Basic_Emoji",
  "Emoji_Keycap_Sequence",
  "RGI_Emoji",
  "RGI_Emoji_Flag_Sequence",
  "RGI_Emoji_Modifier_Sequence",
  "RGI_Emoji_Tag_Sequence",
  "RGI_Emoji_ZWJ_Sequence"
]);
var JsUnicodePropertiesOfStringsMap = /* @__PURE__ */ new Map();
for (const p of JsUnicodePropertiesOfStrings) {
  JsUnicodePropertiesOfStringsMap.set(slug(p), p);
}
var JsUnicodePropertiesPostEs2018 = new Set(
  // ES2019 scripts
  "Dogr Dogra Gong Gunjala_Gondi Hanifi_Rohingya Maka Makasar Medefaidrin Medf Old_Sogdian Rohg Sogd Sogdian Sogo Extended_Pictographic Elym Elymaic Hmnp Nand Nandinagari Nyiakeng_Puachue_Hmong Wancho Wcho Chorasmian Chrs Diak Dives_Akuru Khitan_Small_Script Kits Yezi Yezidi EBase EComp EMod EPres ExtPict Cpmn Cypro_Minoan Old_Uyghur Ougr Tangsa Tnsa Toto Vith Vithkuqi Gara Garay Gukh Gurung_Khema Hrkt Katakana_Or_Hiragana Kawi Kirat_Rai Krai Nag_Mundari Nagm Ol_Onal Onao Sunu Sunuwar Todhri Todr Tulu_Tigalari Tutg Unknown Zzzz".split(" ")
);
var LowerToAlternativeUpperCaseMap = /* @__PURE__ */ new Map([
  [cp(223), cp(7838)],
  // ß, ẞ
  [cp(107), cp(8490)],
  // k, K (Kelvin)
  [cp(229), cp(8491)],
  // å, Å (Angstrom)
  [cp(969), cp(8486)]
  // ω, Ω (Ohm)
]);
var LowerToTitleCaseMap = new Map([
  titleEntry(453),
  titleEntry(456),
  titleEntry(459),
  titleEntry(498),
  ...titleRange(8072, 8079),
  ...titleRange(8088, 8095),
  ...titleRange(8104, 8111),
  titleEntry(8124),
  titleEntry(8140),
  titleEntry(8188)
]);
var PosixClassesMap = /* @__PURE__ */ new Map([
  ["alnum", r2`[\p{Alpha}\p{Nd}]`],
  ["alpha", r2`\p{Alpha}`],
  ["ascii", r2`\p{ASCII}`],
  ["blank", r2`[\p{Zs}\t]`],
  ["cntrl", r2`\p{cntrl}`],
  ["digit", r2`\p{Nd}`],
  ["graph", r2`[\P{space}&&\P{cntrl}&&\P{Cn}&&\P{Cs}]`],
  ["lower", r2`\p{Lower}`],
  ["print", r2`[[\P{space}&&\P{cntrl}&&\P{Cn}&&\P{Cs}]\p{Zs}]`],
  ["punct", r2`[\p{P}\p{S}]`],
  // New value from Oniguruma 6.9.9
  ["space", r2`\p{space}`],
  ["upper", r2`\p{Upper}`],
  ["word", r2`[\p{Alpha}\p{M}\p{Nd}\p{Pc}]`],
  ["xdigit", r2`\p{AHex}`]
]);
var PosixProperties = /* @__PURE__ */ new Set([
  "alnum",
  "blank",
  "graph",
  "print",
  "word",
  "xdigit"
  // The following are available with the same name in JS (see `JsUnicodeProperties`)
  // - alpha (JS: Alpha)
  // - ascii (JS: ASCII)
  // - cntrl (JS: cntrl)
  // - digit (JS: digit)
  // - lower (JS: Lower)
  // - punct (JS: punct)
  // - space (JS: space)
  // - upper (JS: Upper)
]);
function range(start, end) {
  const range2 = [];
  for (let i = start; i <= end; i++) {
    range2.push(i);
  }
  return range2;
}
function slug(name) {
  return name.replace(/[- _]+/g, "").toLowerCase();
}
function titleEntry(codePoint) {
  const char = cp(codePoint);
  return [char.toLowerCase(), char];
}
function titleRange(start, end) {
  return range(start, end).map((codePoint) => titleEntry(codePoint));
}
var UnicodePropertiesWithSpecificCase = /* @__PURE__ */ new Set([
  "Lower",
  "Lowercase",
  "Upper",
  "Uppercase",
  "Ll",
  "Lowercase_Letter",
  "Lt",
  "Titlecase_Letter",
  "Lu",
  "Uppercase_Letter"
  // The `Changes_When_*` properties (and their aliases) could be included, but they're very rare.
  // Some other properties include a handful of chars with specific cases only, but these chars are
  // generally extreme edge cases and using such properties case insensitively generally produces
  // undesired behavior anyway
]);

// src/tokenize.js
var TokenTypes = (
  /** @type {const} */
  {
    Alternator: "Alternator",
    Assertion: "Assertion",
    Backreference: "Backreference",
    Character: "Character",
    CharacterClassClose: "CharacterClassClose",
    CharacterClassHyphen: "CharacterClassHyphen",
    CharacterClassIntersector: "CharacterClassIntersector",
    CharacterClassOpen: "CharacterClassOpen",
    CharacterSet: "CharacterSet",
    Directive: "Directive",
    GroupClose: "GroupClose",
    GroupOpen: "GroupOpen",
    Subroutine: "Subroutine",
    Quantifier: "Quantifier",
    // These aren't allowed in char classes, so they aren't equivalent to JS `[\q{}]`
    VariableLengthCharacterSet: "VariableLengthCharacterSet",
    // Intermediate representation not included in results
    EscapedNumber: "EscapedNumber"
  }
);
var TokenCharacterSetKinds = {
  any: "any",
  digit: "digit",
  dot: "dot",
  hex: "hex",
  non_newline: "non_newline",
  posix: "posix",
  property: "property",
  space: "space",
  word: "word"
};
var TokenDirectiveKinds = {
  flags: "flags",
  keep: "keep"
};
var TokenGroupKinds = {
  atomic: "atomic",
  capturing: "capturing",
  group: "group",
  lookahead: "lookahead",
  lookbehind: "lookbehind"
};
var EscapeCharCodes = /* @__PURE__ */ new Map([
  ["a", 7],
  // alert/bell (Not available in JS)
  ["b", 8],
  // backspace (only in char classes)
  ["e", 27],
  // escape (Not available in JS)
  ["f", 12],
  // form feed
  ["n", 10],
  // line feed
  ["r", 13],
  // carriage return
  ["t", 9],
  // horizontal tab
  ["v", 11]
  // vertical tab
]);
var controlCharPattern = "c.? | C(?:-.?)?";
var unicodePropertyPattern = r2`[pP]\{(?:\^?[\x20\w]+\})?`;
var encodedByteValuePattern = r2`x[89A-Fa-f]\p{AHex}(?:\\x[89A-Fa-f]\p{AHex})*`;
var hexCharPattern = r2`u(?:\p{AHex}{4})? | x\{[^\}]*\}? | x\p{AHex}{0,2}`;
var escapedNumPattern = r2`\d{1,3}`;
var charClassOpenPattern = r2`\[\^?\]?`;
var quantifierRe = /[?*+][?+]?|\{(?:\d+(?:,\d*)?|,\d+)\}\??/;
var tokenRe = new RegExp(r2`
  \\ (?:
    ${controlCharPattern}
    | ${unicodePropertyPattern}
    | ${encodedByteValuePattern}
    | ${hexCharPattern}
    | ${escapedNumPattern}
    | [gk]<[^>]*>?
    | [gk]'[^']*'?
    | .
  )
  | \( (?: \? (?:
    [:=!>(~]
    | <[=!]
    | <[^>]*>
    | '[^']*'
    | # (?:[^)\\] | \\.?)*
    | [imx\-]+[:)]
  )?)?
  | ${quantifierRe.source}
  | ${charClassOpenPattern}
  | .
`.replace(/\s+/g, ""), "gsu");
var charClassTokenRe = new RegExp(r2`
  \\ (?:
    ${controlCharPattern}
    | ${unicodePropertyPattern}
    | ${encodedByteValuePattern}
    | ${hexCharPattern}
    | ${escapedNumPattern}
    | .
  )
  | \[:[^:]*:\]
  | ${charClassOpenPattern}
  | &&
  | .
`.replace(/\s+/g, ""), "gsu");
function tokenize(pattern2, flags = "") {
  if (typeof pattern2 !== "string") {
    throw new Error("String expected as pattern");
  }
  if (!/^[imx]*$/.test(flags)) {
    throw new Error(`Flags "${flags}" unsupported in Oniguruma`);
  }
  const xStack = [flags.includes("x")];
  const context = {
    getCurrentModX: () => xStack.at(-1),
    numOpenGroups: 0,
    popModX() {
      xStack.pop();
    },
    pushModX(isXOn) {
      xStack.push(isXOn);
    },
    replaceCurrentModX(isXOn) {
      xStack[xStack.length - 1] = isXOn;
    }
  };
  let tokens = [];
  let match;
  tokenRe.lastIndex = 0;
  while (match = tokenRe.exec(pattern2)) {
    const result = getTokenWithDetails(context, pattern2, match[0], tokenRe.lastIndex);
    if (result.tokens) {
      tokens.push(...result.tokens);
    } else if (result.token) {
      tokens.push(result.token);
    }
    if (result.lastIndex !== void 0) {
      tokenRe.lastIndex = result.lastIndex;
    }
  }
  const potentialUnnamedCaptureTokens = [];
  let numNamedCaptures = 0;
  tokens.forEach((t) => {
    if (t.type === TokenTypes.GroupOpen) {
      if (t.kind === TokenGroupKinds.capturing) {
        numNamedCaptures++;
        t.number = numNamedCaptures;
      } else if (t.raw === "(") {
        potentialUnnamedCaptureTokens.push(t);
      }
    }
  });
  if (!numNamedCaptures) {
    potentialUnnamedCaptureTokens.forEach((t, i) => {
      t.kind = TokenGroupKinds.capturing;
      t.number = i + 1;
    });
  }
  const numCaptures = numNamedCaptures || potentialUnnamedCaptureTokens.length;
  tokens = tokens.map(
    (t) => t.type === TokenTypes.EscapedNumber ? splitEscapedNumToken(t, numCaptures) : t
  ).flat();
  return {
    tokens,
    flags: {
      ignoreCase: flags.includes("i"),
      // Onig flag m is equivalent to JS flag s
      dotAll: flags.includes("m"),
      // Flag x is fully handled during tokenization
      extended: flags.includes("x")
    }
  };
}
function getTokenWithDetails(context, pattern2, m, lastIndex) {
  const [m0, m1, m2] = m;
  if (m0 === "[") {
    const result = getAllTokensForCharClass(pattern2, m, lastIndex);
    return {
      // Array of all of the char class's tokens
      tokens: result.tokens,
      // Jump forward to the end of the char class
      lastIndex: result.lastIndex
    };
  }
  if (m0 === "\\") {
    if ("AbBGzZ".includes(m1)) {
      return {
        token: createToken(TokenTypes.Assertion, m, {
          kind: m
        })
      };
    }
    if (/^\\g[<']/.test(m)) {
      if (!/^\\g(?:<[^>]+>|'[^']+')$/.test(m)) {
        throw new Error(`Invalid group name "${m}"`);
      }
      return {
        token: createToken(TokenTypes.Subroutine, m)
      };
    }
    if (/^\\k[<']/.test(m)) {
      if (!/^\\k(?:<[^>]+>|'[^']+')$/.test(m)) {
        throw new Error(`Invalid group name "${m}"`);
      }
      return {
        token: createToken(TokenTypes.Backreference, m)
      };
    }
    if (m1 === "K") {
      return {
        token: createToken(TokenTypes.Directive, m, {
          kind: TokenDirectiveKinds.keep
        })
      };
    }
    if (m1 === "N") {
      return {
        token: createToken(TokenTypes.CharacterSet, m, {
          kind: TokenCharacterSetKinds.non_newline
        })
      };
    }
    if (m1 === "O") {
      return {
        token: createToken(TokenTypes.CharacterSet, m, {
          kind: TokenCharacterSetKinds.any
        })
      };
    }
    if ("RX".includes(m1)) {
      return {
        token: createToken(TokenTypes.VariableLengthCharacterSet, m, {
          kind: m
        })
      };
    }
    const result = createTokenForSharedEscape(m, { inCharClass: false });
    return Array.isArray(result) ? { tokens: result } : { token: result };
  }
  if (m0 === "(") {
    if (m2 === "#") {
      if (pattern2[lastIndex] !== ")") {
        throw new Error('Unclosed comment group "(?#"');
      }
      return {
        lastIndex: lastIndex + 1
      };
    }
    if ("-imx".includes(m2)) {
      return {
        token: createTokenForFlagMod(m, context)
      };
    }
    context.pushModX(context.getCurrentModX());
    context.numOpenGroups++;
    if (
      // Unnamed capture if no named captures, else noncapturing group
      m === "(" || // Noncapturing group
      m === "(?:"
    ) {
      return {
        token: createToken(TokenTypes.GroupOpen, m, {
          // For `(`, will later change to `capturing` and add `number` prop if no named captures
          kind: TokenGroupKinds.group
        })
      };
    }
    if (m === "(?>") {
      return {
        token: createToken(TokenTypes.GroupOpen, m, {
          kind: TokenGroupKinds.atomic
        })
      };
    }
    if (m === "(?=" || m === "(?!" || m === "(?<=" || m === "(?<!") {
      return {
        token: createToken(TokenTypes.GroupOpen, m, {
          kind: m2 === "<" ? TokenGroupKinds.lookbehind : TokenGroupKinds.lookahead,
          negate: m.endsWith("!")
        })
      };
    }
    if (m2 === "<" || m2 === "'") {
      return {
        token: createToken(TokenTypes.GroupOpen, m, {
          kind: TokenGroupKinds.capturing,
          name: m.slice(3, -1)
          // Will add `number` in a second pass
        })
      };
    }
    if (m2 === "(") {
      throw new Error(`Unsupported conditional "${m}"`);
    }
    if (m2 === "~") {
      throw new Error(`Unsupported absence operator "${m}"`);
    }
    if (m === "(?") {
      throw new Error("Invalid group");
    }
    throw new Error(`Unexpected group "${m}"`);
  }
  if (m === ")") {
    context.popModX();
    context.numOpenGroups--;
    if (context.numOpenGroups < 0) {
      throw new Error('Unmatched ")"');
    }
    return {
      token: createToken(TokenTypes.GroupClose, m)
    };
  }
  if (m === "#" && context.getCurrentModX()) {
    const end = pattern2.indexOf("\n", lastIndex);
    return {
      // Jump forward to the end of the comment
      lastIndex: end === -1 ? pattern2.length : end
    };
  }
  if (/^\s$/.test(m) && context.getCurrentModX()) {
    const re = /\s+/y;
    re.lastIndex = lastIndex;
    const rest = re.exec(pattern2);
    return {
      // Jump forward to the end of the whitespace
      lastIndex: rest ? re.lastIndex : lastIndex
    };
  }
  if (m === ".") {
    return {
      token: createToken(TokenTypes.CharacterSet, m, {
        kind: TokenCharacterSetKinds.dot
      })
    };
  }
  if (m === "^" || m === "$") {
    return {
      token: createToken(TokenTypes.Assertion, m, {
        kind: m
      })
    };
  }
  if (m === "|") {
    return {
      token: createToken(TokenTypes.Alternator, m)
    };
  }
  if (quantifierRe.test(m)) {
    return {
      token: createTokenForQuantifier(m)
    };
  }
  assertSingleCodePoint(m);
  return {
    token: createToken(TokenTypes.Character, m, {
      value: m.codePointAt(0)
    })
  };
}
function getAllTokensForCharClass(pattern2, opener, lastIndex) {
  assertNonEmptyCharClass(opener);
  const tokens = [createToken(TokenTypes.CharacterClassOpen, opener, {
    negate: opener[1] === "^"
  })];
  let numCharClassesOpen = 1;
  let match;
  charClassTokenRe.lastIndex = lastIndex;
  while (match = charClassTokenRe.exec(pattern2)) {
    const m = match[0];
    if (m[0] === "[" && m[1] !== ":") {
      assertNonEmptyCharClass(m);
      numCharClassesOpen++;
      tokens.push(createToken(TokenTypes.CharacterClassOpen, m, {
        negate: m[1] === "^"
      }));
    } else if (m === "]") {
      numCharClassesOpen--;
      tokens.push(createToken(TokenTypes.CharacterClassClose, m));
      if (!numCharClassesOpen) {
        break;
      }
    } else {
      const result = createTokenForAnyTokenWithinCharClass(m);
      if (Array.isArray(result)) {
        tokens.push(...result);
      } else {
        tokens.push(result);
      }
    }
  }
  return {
    tokens,
    lastIndex: charClassTokenRe.lastIndex || pattern2.length
  };
}
function createTokenForAnyTokenWithinCharClass(raw) {
  if (raw[0] === "\\") {
    return createTokenForSharedEscape(raw, { inCharClass: true });
  }
  if (raw[0] === "[") {
    const posix = /\[:(?<negate>\^?)(?<name>[a-z]+):\]/.exec(raw);
    if (!posix || !PosixClassesMap.get(posix.groups.name)) {
      throw new Error(`Invalid POSIX class "${raw}"`);
    }
    return createToken(TokenTypes.CharacterSet, raw, {
      kind: TokenCharacterSetKinds.posix,
      negate: !!posix.groups.negate,
      value: posix.groups.name
    });
  }
  if (raw === "-") {
    return createToken(TokenTypes.CharacterClassHyphen, raw);
  }
  if (raw === "&&") {
    return createToken(TokenTypes.CharacterClassIntersector, raw);
  }
  assertSingleCodePoint(raw);
  return createToken(TokenTypes.Character, raw, {
    value: raw.codePointAt(0)
  });
}
function createTokenForSharedEscape(raw, { inCharClass }) {
  const char1 = raw[1];
  if (char1 === "c" || char1 === "C") {
    return createTokenForControlChar(raw);
  }
  if ("dDhHsSwW".includes(char1)) {
    return createTokenForShorthandCharClass(raw);
  }
  if (/^\\[pP]\{/.test(raw)) {
    if (raw.length === 3) {
      throw new Error("Incomplete or invalid Unicode property");
    }
    return createTokenForUnicodeProperty(raw);
  }
  if (/^\\x[89A-Fa-f]\p{AHex}/u.test(raw)) {
    try {
      const bytes = raw.split(/\\x/).slice(1).map((hex) => parseInt(hex, 16));
      const decoded = new TextDecoder("utf-8", {
        ignoreBOM: true,
        fatal: true
      }).decode(new Uint8Array(bytes));
      const encoder = new TextEncoder();
      const tokens = [...decoded].map((char) => {
        const raw2 = [...encoder.encode(char)].map((byte) => `\\x${byte.toString(16)}`).join("");
        return createToken(TokenTypes.Character, raw2, {
          value: char.codePointAt(0)
        });
      });
      return tokens;
    } catch (err) {
      throw new Error(`Too short or invalid multibyte code "${raw}"`);
    }
  }
  if (char1 === "u" || char1 === "x") {
    return createToken(TokenTypes.Character, raw, {
      value: getValidatedHexCharCode(raw)
    });
  }
  if (EscapeCharCodes.has(char1)) {
    return createToken(TokenTypes.Character, raw, {
      value: EscapeCharCodes.get(char1)
    });
  }
  if (/\d/.test(char1)) {
    return createToken(TokenTypes.EscapedNumber, raw, {
      inCharClass
    });
  }
  if (raw === "\\") {
    throw new Error(r2`Incomplete escape "\"`);
  }
  if (char1 === "M") {
    throw new Error(`Unsupported meta "${raw}"`);
  }
  if ([...raw].length === 2) {
    return createToken(TokenTypes.Character, raw, {
      value: raw.codePointAt(1)
    });
  }
  throw new Error(`Unexpected escape "${raw}"`);
}
function createToken(type, raw, data) {
  return {
    type,
    raw,
    ...data
  };
}
function createTokenForControlChar(raw) {
  const char = raw[1] === "c" ? raw[2] : raw[3];
  if (!char || !/[A-Za-z]/.test(char)) {
    throw new Error(`Unsupported control character "${raw}"`);
  }
  return createToken(TokenTypes.Character, raw, {
    value: char.toUpperCase().codePointAt(0) - 64
  });
}
function createTokenForFlagMod(raw, context) {
  let { on, off } = /^\(\?(?<on>[imx]*)(?:-(?<off>[imx\-]*))?/.exec(raw).groups;
  off ??= "";
  const isXOn = (context.getCurrentModX() || on.includes("x")) && !off.includes("x");
  const enabledFlags = getFlagPropsForToken(on);
  const disabledFlags = getFlagPropsForToken(off);
  const flagChanges = {};
  enabledFlags && (flagChanges.enable = enabledFlags);
  disabledFlags && (flagChanges.disable = disabledFlags);
  if (raw.endsWith(")")) {
    context.replaceCurrentModX(isXOn);
    return createToken(TokenTypes.Directive, raw, {
      kind: TokenDirectiveKinds.flags,
      flags: flagChanges
    });
  }
  if (raw.endsWith(":")) {
    context.pushModX(isXOn);
    context.numOpenGroups++;
    const token2 = createToken(TokenTypes.GroupOpen, raw, {
      kind: TokenGroupKinds.group
    });
    if (enabledFlags || disabledFlags) {
      token2.flags = flagChanges;
    }
    return token2;
  }
  throw new Error(`Unexpected flag modifier "${raw}"`);
}
function createTokenForQuantifier(raw) {
  const data = {};
  if (raw[0] === "{") {
    const { min, max } = /^\{(?<min>\d*)(?:,(?<max>\d*))?/.exec(raw).groups;
    const limit = 1e5;
    if (+min > limit || +max > limit) {
      throw new Error("Quantifier value unsupported in Oniguruma");
    }
    data.min = +min;
    data.max = max === void 0 ? +min : max === "" ? Infinity : +max;
    data.greedy = !raw.endsWith("?");
    data.possessive = false;
  } else {
    data.min = raw[0] === "+" ? 1 : 0;
    data.max = raw[0] === "?" ? 1 : Infinity;
    data.greedy = raw[1] !== "?";
    data.possessive = raw[1] === "+";
  }
  return createToken(TokenTypes.Quantifier, raw, data);
}
function createTokenForShorthandCharClass(raw) {
  const lower = raw[1].toLowerCase();
  return createToken(TokenTypes.CharacterSet, raw, {
    kind: {
      "d": TokenCharacterSetKinds.digit,
      "h": TokenCharacterSetKinds.hex,
      // Not available in JS
      "s": TokenCharacterSetKinds.space,
      // Different than JS
      "w": TokenCharacterSetKinds.word
    }[lower],
    negate: raw[1] !== lower
  });
}
function createTokenForUnicodeProperty(raw) {
  const { p, neg, value } = /^\\(?<p>[pP])\{(?<neg>\^?)(?<value>[ \w]+)/.exec(raw).groups;
  const negate = p === "P" && !neg || p === "p" && !!neg;
  return createToken(TokenTypes.CharacterSet, raw, {
    kind: TokenCharacterSetKinds.property,
    negate,
    value
  });
}
function getFlagPropsForToken(flags) {
  const obj = {};
  if (flags.includes("i")) {
    obj.ignoreCase = true;
  }
  if (flags.includes("m")) {
    obj.dotAll = true;
  }
  if (flags.includes("x")) {
    obj.extended = true;
  }
  return Object.keys(obj).length ? obj : null;
}
function getValidatedHexCharCode(raw) {
  if (/^(?:\\u(?!\p{AHex}{4})|\\x(?!\p{AHex}{1,2}|\{\p{AHex}{1,8}\}))/u.test(raw)) {
    throw new Error(`Incomplete or invalid escape "${raw}"`);
  }
  const hex = raw[2] === "{" ? /^\\x\{\s*(?<hex>\p{AHex}+)/u.exec(raw).groups.hex : raw.slice(2);
  const dec = parseInt(hex, 16);
  if (dec > 127 && /^\\x\p{AHex}/u.test(raw)) {
    throw new Error(r2`\xNN above 7F unsupported in Oniguruma "${raw}"`);
  } else if (dec > 1310719) {
    throw new Error(`Invalid out of range "${raw}"`);
  } else if (dec > 1114111) {
    throw new Error(`Invalid out of range in JS "${raw}"`);
  }
  return dec;
}
function splitEscapedNumToken(token2, numCaptures) {
  const { raw, inCharClass } = token2;
  const value = raw.slice(1);
  if (!inCharClass && // Single digit 1-9 outside a char class is always treated as a backref
  (value !== "0" && value.length === 1 || // Leading 0 makes it octal; backrefs can't include following literal digits
  value[0] !== "0" && +value <= numCaptures)) {
    return [createToken(TokenTypes.Backreference, raw)];
  }
  const tokens = [];
  const matches = value.match(/^[0-7]+|\d/g);
  for (let i = 0; i < matches.length; i++) {
    const m = matches[i];
    const value2 = i === 0 && m !== "8" && m !== "9" ? parseInt(m, 8) : m.codePointAt(0);
    tokens.push(createToken(TokenTypes.Character, (i === 0 ? "\\" : "") + m, {
      value: value2
    }));
  }
  return tokens;
}
function assertNonEmptyCharClass(raw) {
  if (raw.endsWith("]")) {
    throw new Error(`Empty character class "${raw}" unsupported in Oniguruma`);
  }
}
function assertSingleCodePoint(raw) {
  if ([...raw].length !== 1) {
    throw new Error(`Expected "${raw}" to be a single code point`);
  }
}

// src/traverse.js
function traverse(path, state, visitor) {
  let ast = path.node;
  while (ast.parent) {
    ast = ast.parent;
  }
  function traverseArray(array, parent) {
    for (let i = 0; i < array.length; i++) {
      const keyShift = traverseNode(array[i], parent, i, array);
      i = Math.max(-1, i + keyShift);
    }
  }
  function traverseNode(node, parent = null, key = null, container = null) {
    let keyShift = 0;
    let skipTraversingKidsOfPath = false;
    const path2 = {
      node,
      parent,
      key,
      container,
      ast,
      remove() {
        throwIfNot(container, "Container expected").splice(Math.max(0, key + keyShift), 1);
        keyShift -= 1;
      },
      removeAllNextSiblings() {
        return throwIfNot(container, "Container expected").splice(key + 1);
      },
      removeAllPrevSiblings() {
        const shifted = key + keyShift;
        keyShift -= shifted;
        return throwIfNot(container, "Container expected").splice(0, Math.max(0, shifted));
      },
      replaceWith(newNode) {
        setParent(newNode, parent);
        if (container) {
          container[Math.max(0, key + keyShift)] = newNode;
        } else {
          parent[key] = newNode;
        }
      },
      skip() {
        skipTraversingKidsOfPath = true;
      }
    };
    const visitorKey = getAstTypeAliases(node).find((key2) => !!visitor[key2]);
    const methods = visitorKey && visitor[visitorKey];
    const enterFn = typeof methods === "function" ? methods : methods?.enter;
    const exitFn = methods?.exit;
    enterFn?.(path2, state);
    if (!skipTraversingKidsOfPath) {
      switch (node.type) {
        case AstTypes.Regex:
          traverseNode(node.pattern, node, "pattern");
          traverseNode(node.flags, node, "flags");
          break;
        case AstTypes.Alternative:
        case AstTypes.CharacterClass:
          traverseArray(node.elements, node);
          break;
        case AstTypes.Assertion:
          if (isLookaround(node)) {
            traverseArray(node.alternatives, node);
          }
          break;
        case AstTypes.Backreference:
        case AstTypes.Character:
        case AstTypes.CharacterSet:
        case AstTypes.Directive:
        case AstTypes.Flags:
        case AstTypes.Recursion:
        case AstTypes.Subroutine:
        case AstTypes.VariableLengthCharacterSet:
          break;
        case AstTypes.CapturingGroup:
        case AstTypes.Group:
        case AstTypes.Pattern:
          traverseArray(node.alternatives, node);
          break;
        case AstTypes.CharacterClassIntersection:
          traverseArray(node.classes, node);
          break;
        case AstTypes.CharacterClassRange:
          traverseNode(node.min, node, "min");
          traverseNode(node.max, node, "max");
          break;
        case AstTypes.Quantifier:
          traverseNode(node.element, node, "element");
          break;
        default:
          throw new Error(`Unexpected node type "${node.type}"`);
      }
    }
    exitFn?.(path2, state);
    return keyShift;
  }
  traverseNode(path.node, path.parent, path.key, path.container);
}
function setParent(node, parent) {
  if ("parent" in parent) {
    node.parent = parent;
  }
}

// src/parse.js
var AstTypes = {
  Alternative: "Alternative",
  Assertion: "Assertion",
  Backreference: "Backreference",
  CapturingGroup: "CapturingGroup",
  Character: "Character",
  CharacterClass: "CharacterClass",
  CharacterClassIntersection: "CharacterClassIntersection",
  CharacterClassRange: "CharacterClassRange",
  CharacterSet: "CharacterSet",
  Directive: "Directive",
  Flags: "Flags",
  Group: "Group",
  Pattern: "Pattern",
  Quantifier: "Quantifier",
  Regex: "Regex",
  Subroutine: "Subroutine",
  VariableLengthCharacterSet: "VariableLengthCharacterSet",
  // Used only by the transformer for Regex+ ASTs
  Recursion: "Recursion"
};
var AstTypeAliases = {
  AnyGroup: "AnyGroup",
  AnyNode: "AnyNode"
};
function getAstTypeAliases(node) {
  const { type } = node;
  const types = [AstTypeAliases.AnyNode];
  if (isLookaround(node) || type === AstTypes.CapturingGroup || type === AstTypes.Group) {
    types.push(AstTypeAliases.AnyGroup);
  }
  types.push(type);
  return types;
}
var AstAssertionKinds = {
  line_end: "line_end",
  line_start: "line_start",
  lookahead: "lookahead",
  lookbehind: "lookbehind",
  search_start: "search_start",
  string_end: "string_end",
  string_end_newline: "string_end_newline",
  string_start: "string_start",
  word_boundary: "word_boundary"
};
var AstCharacterSetKinds = TokenCharacterSetKinds;
var AstDirectiveKinds = TokenDirectiveKinds;
var AstVariableLengthCharacterSetKinds = {
  grapheme: "grapheme",
  newline: "newline"
};
function parse({ tokens, flags }, options) {
  const opts = {
    skipBackrefValidation: false,
    skipPropertyNameValidation: false,
    verbose: false,
    ...options
  };
  const context = {
    capturingGroups: [],
    current: 0,
    hasNumberedRef: false,
    namedGroupsByName: /* @__PURE__ */ new Map(),
    parent: null,
    skipBackrefValidation: opts.skipBackrefValidation,
    skipPropertyNameValidation: opts.skipPropertyNameValidation,
    subroutines: [],
    token: null,
    tokens,
    verbose: opts.verbose,
    walk
  };
  function walk(parent, state) {
    const token2 = tokens[context.current];
    context.parent = parent;
    context.token = token2;
    context.current++;
    switch (token2.type) {
      case TokenTypes.Alternator:
        return createAlternative();
      case TokenTypes.Assertion:
        return createAssertionFromToken(token2);
      case TokenTypes.Backreference:
        return parseBackreference(context);
      case TokenTypes.Character:
        return createCharacter(token2.value);
      case TokenTypes.CharacterClassHyphen:
        return parseCharacterClassHyphen(context, state);
      case TokenTypes.CharacterClassOpen:
        return parseCharacterClassOpen(context, state);
      case TokenTypes.CharacterSet:
        return parseCharacterSet(context);
      case TokenTypes.Directive:
        return createDirectiveFromToken(token2);
      case TokenTypes.GroupOpen:
        return parseGroupOpen(context, state);
      case TokenTypes.Quantifier:
        return parseQuantifier(context);
      case TokenTypes.Subroutine:
        return parseSubroutine(context);
      case TokenTypes.VariableLengthCharacterSet:
        return createVariableLengthCharacterSet(token2.kind);
      default:
        throw new Error(`Unexpected token type "${token2.type}"`);
    }
  }
  const ast = createRegex(createPattern(), createFlags(flags));
  let top = ast.pattern.alternatives[0];
  while (context.current < tokens.length) {
    const node = walk(top, {});
    if (node.type === AstTypes.Alternative) {
      ast.pattern.alternatives.push(node);
      top = node;
    } else {
      top.elements.push(node);
    }
  }
  const { capturingGroups, hasNumberedRef, namedGroupsByName, subroutines } = context;
  if (hasNumberedRef && namedGroupsByName.size) {
    throw new Error("Numbered backref/subroutine not allowed when using named capture");
  }
  for (const { ref } of subroutines) {
    if (typeof ref === "number") {
      if (ref > capturingGroups.length) {
        throw new Error(`Subroutine uses a group number that's not defined`);
      }
    } else if (!namedGroupsByName.has(ref)) {
      throw new Error(r2`Subroutine uses a group name that's not defined "\g<${ref}>"`);
    } else if (namedGroupsByName.get(ref).length > 1) {
      throw new Error(r2`Subroutine uses a duplicate group name "\g<${ref}>"`);
    }
  }
  traverse({ node: ast }, null, {
    AnyNode({ node, parent }) {
      node.parent = parent;
    }
  });
  return ast;
}
function parseBackreference(context) {
  const { raw } = context.token;
  const hasKWrapper = /^\\k[<']/.test(raw);
  const ref = hasKWrapper ? raw.slice(3, -1) : raw.slice(1);
  const fromNum = (num, isRelative = false) => {
    const numCapturesToLeft = context.capturingGroups.length;
    let orphan = false;
    if (num > numCapturesToLeft) {
      if (context.skipBackrefValidation) {
        orphan = true;
      } else {
        throw new Error(`Not enough capturing groups defined to the left "${raw}"`);
      }
    }
    context.hasNumberedRef = true;
    return createBackreference(isRelative ? numCapturesToLeft + 1 - num : num, { orphan });
  };
  if (hasKWrapper) {
    const numberedRef = /^(?<sign>-?)0*(?<num>[1-9]\d*)$/.exec(ref);
    if (numberedRef) {
      return fromNum(+numberedRef.groups.num, !!numberedRef.groups.sign);
    }
    if (/[-+]/.test(ref)) {
      throw new Error(`Invalid backref name "${raw}"`);
    }
    if (!context.namedGroupsByName.has(ref)) {
      throw new Error(`Group name not defined to the left "${raw}"`);
    }
    return createBackreference(ref);
  }
  return fromNum(+ref);
}
function parseCharacterClassHyphen(context, state) {
  const { parent, tokens, walk } = context;
  const prevSiblingNode = parent.elements.at(-1);
  const nextToken = tokens[context.current];
  if (prevSiblingNode && prevSiblingNode.type !== AstTypes.CharacterClass && prevSiblingNode.type !== AstTypes.CharacterClassRange && nextToken && nextToken.type !== TokenTypes.CharacterClassOpen && nextToken.type !== TokenTypes.CharacterClassClose && nextToken.type !== TokenTypes.CharacterClassIntersector) {
    const nextNode = walk(parent, state);
    if (prevSiblingNode.type === AstTypes.Character && nextNode.type === AstTypes.Character) {
      parent.elements.pop();
      return createCharacterClassRange(prevSiblingNode, nextNode);
    }
    throw new Error("Invalid character class range");
  }
  return createCharacter(45);
}
function parseCharacterClassOpen(context, state) {
  const { token: token2, tokens, verbose, walk } = context;
  let node = createCharacterClass({ negate: token2.negate });
  const intersection = node.elements[0];
  let nextToken = throwIfUnclosedCharacterClass(tokens[context.current]);
  while (nextToken.type !== TokenTypes.CharacterClassClose) {
    if (nextToken.type === TokenTypes.CharacterClassIntersector) {
      intersection.classes.push(createCharacterClass({ negate: false, baseOnly: true }));
      context.current++;
    } else {
      const cc = intersection.classes.at(-1);
      cc.elements.push(walk(cc, state));
    }
    nextToken = throwIfUnclosedCharacterClass(tokens[context.current]);
  }
  if (!verbose) {
    optimizeCharacterClassIntersection(intersection);
  }
  if (intersection.classes.length === 1) {
    const cc = intersection.classes[0];
    cc.negate = node.negate !== cc.negate;
    node = cc;
  }
  context.current++;
  return node;
}
function parseCharacterSet({ token: token2, skipPropertyNameValidation }) {
  let { kind, negate, value } = token2;
  if (kind === TokenCharacterSetKinds.property) {
    const normalized = slug(value);
    if (PosixProperties.has(normalized)) {
      kind = TokenCharacterSetKinds.posix;
      value = normalized;
    } else {
      return createUnicodeProperty(value, {
        negate,
        skipPropertyNameValidation
      });
    }
  }
  const node = {
    type: AstTypes.CharacterSet,
    kind: throwIfNot(AstCharacterSetKinds[kind], `Unexpected character set kind "${kind}"`)
  };
  if (kind === TokenCharacterSetKinds.digit || kind === TokenCharacterSetKinds.hex || kind === TokenCharacterSetKinds.posix || kind === TokenCharacterSetKinds.space || kind === TokenCharacterSetKinds.word) {
    node.negate = negate;
    if (kind === TokenCharacterSetKinds.posix) {
      node.value = value;
    }
  }
  return node;
}
function parseGroupOpen(context, state) {
  const { token: token2, tokens, capturingGroups, namedGroupsByName, verbose, walk } = context;
  let node = createByGroupKind(token2);
  if (node.type === AstTypes.CapturingGroup) {
    capturingGroups.push(node);
    if (node.name) {
      getOrCreate(namedGroupsByName, node.name, []).push(node);
    }
  }
  let nextToken = throwIfUnclosedGroup(tokens[context.current]);
  while (nextToken.type !== TokenTypes.GroupClose) {
    if (nextToken.type === TokenTypes.Alternator) {
      node.alternatives.push(createAlternative());
      context.current++;
    } else {
      const alt = node.alternatives.at(-1);
      alt.elements.push(walk(alt, state));
    }
    nextToken = throwIfUnclosedGroup(tokens[context.current]);
  }
  if (!verbose) {
    node = getOptimizedGroup(node);
  }
  context.current++;
  return node;
}
function parseQuantifier({ token: token2, parent }) {
  const { min, max, greedy, possessive: possessive2 } = token2;
  const quantifiedNode = parent.elements.at(-1);
  if (!quantifiedNode || quantifiedNode.type === AstTypes.Assertion || quantifiedNode.type === AstTypes.Directive) {
    throw new Error(`Quantifier requires a repeatable token`);
  }
  const node = createQuantifier(quantifiedNode, min, max, greedy, possessive2);
  parent.elements.pop();
  return node;
}
function parseSubroutine(context) {
  const { token: token2, capturingGroups, subroutines } = context;
  let ref = token2.raw.slice(3, -1);
  const numberedRef = /^(?<sign>[-+]?)0*(?<num>[1-9]\d*)$/.exec(ref);
  if (numberedRef) {
    const num = +numberedRef.groups.num;
    const numCapturesToLeft = capturingGroups.length;
    context.hasNumberedRef = true;
    ref = {
      "": num,
      "+": numCapturesToLeft + num,
      "-": numCapturesToLeft + 1 - num
    }[numberedRef.groups.sign];
    if (ref < 1) {
      throw new Error("Invalid subroutine number");
    }
  } else if (ref === "0") {
    ref = 0;
  }
  const node = createSubroutine(ref);
  subroutines.push(node);
  return node;
}
function createAlternative() {
  return {
    type: AstTypes.Alternative,
    elements: []
  };
}
function createAssertionFromToken({ type, kind, negate }) {
  if (type === TokenTypes.GroupOpen) {
    return createLookaround({
      behind: kind === TokenGroupKinds.lookbehind,
      negate
    });
  }
  const nodeKind = throwIfNot({
    "^": AstAssertionKinds.line_start,
    "$": AstAssertionKinds.line_end,
    "\\A": AstAssertionKinds.string_start,
    "\\b": AstAssertionKinds.word_boundary,
    "\\B": AstAssertionKinds.word_boundary,
    "\\G": AstAssertionKinds.search_start,
    "\\z": AstAssertionKinds.string_end,
    "\\Z": AstAssertionKinds.string_end_newline
  }[kind], `Unexpected assertion kind "${kind}"`);
  const node = {
    type: AstTypes.Assertion,
    kind: nodeKind
  };
  if (nodeKind === AstAssertionKinds.word_boundary) {
    node.negate = kind === r2`\B`;
  }
  return node;
}
function createBackreference(ref, { orphan } = {}) {
  return {
    type: AstTypes.Backreference,
    ...orphan && { orphan },
    ref
  };
}
function createByGroupKind(token2) {
  const { kind, number, name, flags } = token2;
  switch (kind) {
    case TokenGroupKinds.atomic:
      return createGroup({ atomic: true });
    case TokenGroupKinds.capturing:
      return createCapturingGroup(number, name);
    case TokenGroupKinds.group:
      return createGroup({ flags });
    case TokenGroupKinds.lookahead:
    case TokenGroupKinds.lookbehind:
      return createAssertionFromToken(token2);
    default:
      throw new Error(`Unexpected group kind "${kind}"`);
  }
}
function createCapturingGroup(number, name) {
  const hasName = name !== void 0;
  if (hasName && !isValidGroupNameOniguruma(name)) {
    throw new Error(`Group name "${name}" invalid in Oniguruma`);
  }
  return {
    type: AstTypes.CapturingGroup,
    number,
    ...hasName && { name },
    alternatives: [createAlternative()]
  };
}
function createCharacter(charCode) {
  return {
    type: AstTypes.Character,
    value: charCode
  };
}
function createCharacterClass(options) {
  const opts = {
    baseOnly: false,
    negate: false,
    ...options
  };
  return {
    type: AstTypes.CharacterClass,
    negate: opts.negate,
    elements: opts.baseOnly ? [] : [createCharacterClassIntersection()]
  };
}
function createCharacterClassIntersection() {
  return {
    type: AstTypes.CharacterClassIntersection,
    classes: [createCharacterClass({ negate: false, baseOnly: true })]
  };
}
function createCharacterClassRange(min, max) {
  if (max.value < min.value) {
    throw new Error("Character class range out of order");
  }
  return {
    type: AstTypes.CharacterClassRange,
    min,
    max
  };
}
function createDirectiveFromToken({ kind, flags }) {
  const node = {
    type: AstTypes.Directive,
    kind: throwIfNot(AstDirectiveKinds[kind], `Unexpected directive kind "${kind}"`)
  };
  if (kind === TokenDirectiveKinds.flags) {
    node.flags = flags;
  }
  return node;
}
function createFlags({ ignoreCase, dotAll, extended }) {
  return {
    type: AstTypes.Flags,
    ignoreCase,
    dotAll,
    extended
  };
}
function createGroup({ atomic: atomic2, flags } = {}) {
  return {
    type: AstTypes.Group,
    ...atomic2 && { atomic: atomic2 },
    ...flags && { flags },
    alternatives: [createAlternative()]
  };
}
function createLookaround({ behind = false, negate = false } = {}) {
  return {
    type: AstTypes.Assertion,
    kind: behind ? AstAssertionKinds.lookbehind : AstAssertionKinds.lookahead,
    negate,
    alternatives: [createAlternative()]
  };
}
function createPattern() {
  return {
    type: AstTypes.Pattern,
    alternatives: [createAlternative()]
  };
}
function createQuantifier(element, min, max, greedy, possessive2) {
  if (max < min) {
    throw new Error("Quantifier range out of order");
  }
  const node = {
    type: AstTypes.Quantifier,
    min,
    max,
    greedy,
    possessive: possessive2,
    element
  };
  return node;
}
function createRegex(pattern2, flags) {
  return {
    type: AstTypes.Regex,
    pattern: pattern2,
    flags
  };
}
function createSubroutine(ref) {
  return {
    type: AstTypes.Subroutine,
    ref
  };
}
function createUnicodeProperty(value, options) {
  const opts = {
    negate: false,
    skipPropertyNameValidation: false,
    ...options
  };
  return {
    type: AstTypes.CharacterSet,
    kind: AstCharacterSetKinds.property,
    value: opts.skipPropertyNameValidation ? value : getJsUnicodePropertyName(value),
    negate: opts.negate
  };
}
function createVariableLengthCharacterSet(kind) {
  return {
    type: AstTypes.VariableLengthCharacterSet,
    kind: throwIfNot({
      "\\R": AstVariableLengthCharacterSetKinds.newline,
      "\\X": AstVariableLengthCharacterSetKinds.grapheme
    }[kind], `Unexpected varcharset kind "${kind}"`)
  };
}
function getJsUnicodePropertyName(value) {
  const slugged = slug(value);
  if (JsUnicodePropertiesOfStringsMap.has(slugged)) {
    throw new Error(r2`Unicode property "\p{${value}}" unsupported in Oniguruma`);
  }
  const jsName = JsUnicodePropertiesMap.get(slugged);
  if (jsName) {
    return jsName;
  }
  return value.trim().replace(/\s+/g, "_").replace(/[A-Z][a-z]+(?=[A-Z])/g, "$&_").replace(/[A-Za-z]+/g, (m) => m[0].toUpperCase() + m.slice(1).toLowerCase());
}
function getOptimizedGroup(node) {
  const firstAlt = node.alternatives[0];
  const firstAltFirstEl = firstAlt.elements[0];
  if (node.type === AstTypes.Group && node.alternatives.length === 1 && firstAlt.elements.length === 1 && firstAltFirstEl.type === AstTypes.Group && !(node.atomic && firstAltFirstEl.flags) && !(node.flags && (firstAltFirstEl.atomic || firstAltFirstEl.flags))) {
    if (node.atomic) {
      firstAltFirstEl.atomic = true;
    } else if (node.flags) {
      firstAltFirstEl.flags = node.flags;
    }
    return firstAltFirstEl;
  }
  return node;
}
function isLookaround({ type, kind }) {
  return type === AstTypes.Assertion && (kind === AstAssertionKinds.lookahead || kind === AstAssertionKinds.lookbehind);
}
function isValidGroupNameOniguruma(name) {
  return !/^(?:[-\d]|$)/.test(name);
}
function optimizeCharacterClassIntersection(intersection) {
  for (let i = 0; i < intersection.classes.length; i++) {
    const cc = intersection.classes[i];
    const firstChild = cc.elements[0];
    if (cc.elements.length === 1 && firstChild.type === AstTypes.CharacterClass) {
      intersection.classes[i] = firstChild;
      firstChild.negate = cc.negate !== firstChild.negate;
    }
  }
}
function throwIfUnclosedCharacterClass(token2) {
  return throwIfNot(token2, "Unclosed character class");
}
function throwIfUnclosedGroup(token2) {
  return throwIfNot(token2, "Unclosed group");
}

// src/subclass-strategies.js
function applySubclassStrategies(ast, accuracy) {
  const alts = ast.pattern.alternatives;
  const firstEl = alts[0].elements[0];
  if (alts.length > 1 || !firstEl) {
    return null;
  }
  const hasWrapperGroup = alts[0].elements.length === 1 && (firstEl.type === AstTypes.CapturingGroup || firstEl.type === AstTypes.Group) && firstEl.alternatives.length === 1;
  const singleAltIn = hasWrapperGroup ? firstEl.alternatives[0] : alts[0];
  const firstElIn = hasWrapperGroup ? singleAltIn.elements[0] : firstEl;
  if (!firstElIn) {
    return null;
  }
  if ((firstElIn.type === AstTypes.CapturingGroup || firstElIn.type === AstTypes.Group) && firstElIn.alternatives.length === 2 && firstElIn.alternatives[0].elements.length === 1 && firstElIn.alternatives[1].elements.length === 1) {
    const el1 = firstElIn.alternatives[0].elements[0];
    const el2 = firstElIn.alternatives[1].elements[0];
    if (el1.kind === AstAssertionKinds.line_start && el2.kind === AstAssertionKinds.search_start || el1.kind === AstAssertionKinds.search_start && el2.kind === AstAssertionKinds.line_start) {
      if (el1.kind === AstAssertionKinds.line_start) {
        firstElIn.alternatives.pop();
      } else {
        firstElIn.alternatives.shift();
      }
      return { name: "line_or_search_start" };
    }
  }
  if (isNegatedSearchStart(firstElIn)) {
    firstElIn.parent.elements.shift();
    return { name: "not_search_start" };
  }
  const negGIndex = singleAltIn.elements.findIndex((el) => isNegatedSearchStart(el));
  if (negGIndex > -1 && singleAltIn.elements.every((el) => el.type === AstTypes.Assertion)) {
    singleAltIn.elements.splice(negGIndex, 1);
    return { name: "not_search_start" };
  }
  if (isLookaround(firstElIn) && !firstElIn.negate && firstElIn.alternatives.length > 1) {
    const siblingAlts = [];
    let hasGAlt = false;
    firstElIn.alternatives.forEach((alt) => {
      if (alt.elements.length === 1 && alt.elements[0].kind === AstAssertionKinds.search_start) {
        hasGAlt = true;
      } else {
        siblingAlts.push(alt);
      }
    });
    if (hasGAlt && siblingAlts.length) {
      let supported = true;
      if (siblingAlts.some((alt) => alt.elements.some((el) => {
        return el.type === AstTypes.CapturingGroup || el.type === AstTypes.Group || el.type === AstTypes.Subroutine || isLookaround(el);
      }))) {
        if (accuracy === "loose") {
          supported = false;
        } else {
          throw new Error(r`Uses "\G" in a way that's unsupported`);
        }
      }
      if (supported) {
        const emulationGroup = adoptAndSwapKids(createGroup(), [
          adoptAndSwapKids(createAlternative(), [createUnicodeProperty("<<", { skipPropertyNameValidation: true })]),
          ...siblingAlts,
          adoptAndSwapKids(createAlternative(), [createUnicodeProperty(">>", { skipPropertyNameValidation: true })])
        ]);
        emulationGroup.parent = firstElIn.parent;
        firstElIn.parent.elements[0] = emulationGroup;
        return { name: "after_search_start_or_subpattern" };
      }
    }
  }
  return null;
}
function isNegatedSearchStart(node) {
  return isLookaround(node) && node.negate && node.alternatives.length === 1 && node.alternatives[0].elements.length === 1 && node.alternatives[0].elements[0].kind === AstAssertionKinds.search_start;
}

// node_modules/emoji-regex-xs/index.mjs
var r3 = String.raw;
var seq = r3`(?:\p{Emoji}\uFE0F\u20E3?|\p{Emoji_Modifier_Base}\p{Emoji_Modifier}?|\p{Emoji_Presentation})`;
var sTags = r3`\u{E0061}-\u{E007A}`;
var emoji_regex_xs_default = () => new RegExp(r3`[\u{1F1E6}-\u{1F1FF}]{2}|\u{1F3F4}[${sTags}]{2}[\u{E0030}-\u{E0039}${sTags}]{1,3}\u{E007F}|${seq}(?:\u200D${seq})*`, "gu");

// src/transform.js
function transform(ast, options) {
  const opts = {
    // A couple edge cases exist where options `accuracy` and `bestEffortTarget` are used:
    // - `VariableLengthCharacterSet` kind `grapheme` (`\X`): An exact representation would require
    //   heavy Unicode data; a best-effort approximation requires knowing the target.
    // - `CharacterSet` kind `posix` with values `graph` and `print`: Their complex exact
    //   representations are hard to change after the fact in the generator to a best-effort
    //   approximation based on the target, so produce the appropriate structure here.
    accuracy: "default",
    avoidSubclass: false,
    bestEffortTarget: "ES2025",
    ...options
  };
  const strategy = opts.avoidSubclass ? null : applySubclassStrategies(ast, opts.accuracy);
  const firstPassState = {
    accuracy: opts.accuracy,
    flagDirectivesByAlt: /* @__PURE__ */ new Map(),
    minTargetEs2024: isMinTarget(opts.bestEffortTarget, "ES2024"),
    // Subroutines can appear before the groups they ref, so collect reffed nodes for a second pass 
    subroutineRefMap: /* @__PURE__ */ new Map(),
    supportedGNodes: /* @__PURE__ */ new Set()
  };
  traverse({ node: ast }, firstPassState, FirstPassVisitor);
  const globalFlags = {
    dotAll: ast.flags.dotAll,
    ignoreCase: ast.flags.ignoreCase
  };
  const secondPassState = {
    currentFlags: globalFlags,
    prevFlags: null,
    globalFlags,
    groupOriginByCopy: /* @__PURE__ */ new Map(),
    groupsWithDuplicateNamesToRemove: /* @__PURE__ */ new Set(),
    multiplexCapturesToLeftByRef: /* @__PURE__ */ new Map(),
    namedGroupsInScopeByAlt: /* @__PURE__ */ new Map(),
    openDirectCaptures: /* @__PURE__ */ new Set(),
    openSubroutineRefs: /* @__PURE__ */ new Set(),
    reffedNodesByBackreference: /* @__PURE__ */ new Map(),
    subroutineRefMap: firstPassState.subroutineRefMap
  };
  traverse({ node: ast }, secondPassState, SecondPassVisitor);
  const thirdPassState = {
    groupsWithDuplicateNamesToRemove: secondPassState.groupsWithDuplicateNamesToRemove,
    highestOrphanBackref: 0,
    numCapturesToLeft: 0,
    reffedNodesByBackreference: secondPassState.reffedNodesByBackreference
  };
  traverse({ node: ast }, thirdPassState, ThirdPassVisitor);
  if (strategy) {
    ast._strategy = strategy;
  }
  return ast;
}
var FirstPassVisitor = {
  Alternative: {
    enter({ node, parent, key }, { flagDirectivesByAlt }) {
      const flagDirectives = node.elements.filter((el) => el.kind === AstDirectiveKinds.flags);
      for (let i = key + 1; i < parent.alternatives.length; i++) {
        const forwardSiblingAlt = parent.alternatives[i];
        getOrCreate(flagDirectivesByAlt, forwardSiblingAlt, []).push(...flagDirectives);
      }
    },
    exit({ node }, { flagDirectivesByAlt }) {
      if (flagDirectivesByAlt.get(node)?.length) {
        const flags = getCombinedFlagModsFromFlagNodes(flagDirectivesByAlt.get(node));
        if (flags) {
          const flagGroup = prepContainer(createGroup({ flags }), node.elements);
          flagGroup.parent = node;
          node.elements = [flagGroup];
        }
      }
    }
  },
  Assertion({ node, ast, remove, replaceWith }, { accuracy, supportedGNodes }) {
    const { kind, negate } = node;
    if (kind === AstAssertionKinds.line_end) {
      replaceWith(parseFragment(r2`(?=\z|\n)`));
    } else if (kind === AstAssertionKinds.line_start) {
      replaceWith(parseFragment(r2`(?<=\A|\n)`));
    } else if (kind === AstAssertionKinds.search_start) {
      if (!supportedGNodes.has(node) && accuracy !== "loose") {
        throw new Error(r2`Uses "\G" in a way that's unsupported`);
      }
      ast.flags.sticky = true;
      remove();
    } else if (kind === AstAssertionKinds.string_end_newline) {
      replaceWith(parseFragment(r2`(?=\n?\z)`));
    } else if (kind === AstAssertionKinds.word_boundary) {
      const wordChar = r2`[\p{L}\p{N}\p{Pc}]`;
      const b = `(?:(?<=${wordChar})(?!${wordChar})|(?<!${wordChar})(?=${wordChar}))`;
      const B = `(?:(?<=${wordChar})(?=${wordChar})|(?<!${wordChar})(?!${wordChar}))`;
      replaceWith(parseFragment(negate ? B : b));
    }
  },
  CapturingGroup({ node }, { subroutineRefMap }) {
    const { name, number } = node;
    if (name && !isValidGroupNameJs(name)) {
      throw new Error(`Group name "${name}" invalid in JS`);
    }
    subroutineRefMap.set(name ?? number, node);
  },
  CharacterSet({ node, replaceWith }, { accuracy, minTargetEs2024 }) {
    const { kind, negate, value } = node;
    if (kind === AstCharacterSetKinds.any) {
      replaceWith(createUnicodeProperty("Any"));
    } else if (kind === AstCharacterSetKinds.hex) {
      replaceWith(createUnicodeProperty("AHex", { negate }));
    } else if (kind === AstCharacterSetKinds.non_newline) {
      replaceWith(parseFragment(r2`[^\n]`));
    } else if (kind === AstCharacterSetKinds.posix) {
      if (!minTargetEs2024 && (value === "graph" || value === "print")) {
        if (accuracy === "strict") {
          throw new Error(`POSIX class "${value}" requires min target ES2024 or non-strict accuracy`);
        }
        let ascii = {
          graph: "!-~",
          print: " -~"
        }[value];
        if (negate) {
          ascii = `\0-${cp(ascii.codePointAt(0) - 1)}${cp(ascii.codePointAt(2) + 1)}-\u{10FFFF}`;
        }
        replaceWith(parseFragment(`[${ascii}]`));
      } else {
        const negateableNode = parseFragment(PosixClassesMap.get(value));
        negateableNode.negate = negate;
        replaceWith(negateableNode);
      }
    } else if (kind === AstCharacterSetKinds.property) {
      if (!JsUnicodeProperties.has(value)) {
        node.key = "sc";
      }
    } else if (kind === AstCharacterSetKinds.space) {
      const s = parseFragment("[ 	\n\v\f\r]");
      s.negate = negate;
      replaceWith(s);
    }
  },
  Directive(path, state) {
    const { node, parent, ast, remove, replaceWith, removeAllPrevSiblings, removeAllNextSiblings } = path;
    const { kind, flags } = node;
    if (kind === AstDirectiveKinds.flags) {
      if (!flags.enable && !flags.disable) {
        remove();
      } else {
        const flagGroup = prepContainer(createGroup({ flags }), removeAllNextSiblings());
        replaceWith(flagGroup);
        traverseReplacement(flagGroup, path, state, FirstPassVisitor);
      }
    } else if (kind === AstDirectiveKinds.keep) {
      if (parent.parent !== ast.pattern || ast.pattern.alternatives.length > 1) {
        throw new Error(r2`Uses "\K" in a way that's unsupported`);
      }
      replaceWith(prepContainer(createLookaround({ behind: true }), removeAllPrevSiblings()));
    }
  },
  Flags({ node, parent }) {
    delete node.extended;
    Object.assign(node, {
      // JS flag g; no Onig equiv
      global: false,
      // JS flag d; no Onig equiv
      hasIndices: false,
      // JS flag m; no Onig equiv but its behavior is always on in Onig. Onig's only line break
      // char is line feed, unlike JS, so this flag isn't used since it would produce inaccurate
      // results (also allows `^` and `$` to be used in the generator for string start and end)
      multiline: false,
      // JS flag y; no Onig equiv, but used for `\G` emulation
      sticky: node.sticky ?? false
      // Note: Regex+ doesn't allow explicitly adding flags it handles implicitly, so leave out
      // properties `unicode` (JS flag u) and `unicodeSets` (JS flag v). Keep the existing values
      // for `ignoreCase` (flag i) and `dotAll` (JS flag s, but Onig flag m)
    });
    parent.options = {
      disable: {
        // Onig uses different rules for flag x than Regex+, so disable the implicit flag
        x: true,
        // Onig has no flag to control "named capture only" mode but contextually applies its
        // behavior when named capturing is used, so disable Regex+'s implicit flag for it
        n: true
      },
      force: {
        // Always add flag v because we're generating an AST that relies on it (it enables JS
        // support for Onig features nested classes, set intersection, Unicode properties, etc.).
        // However, the generator might disable flag v based on its `target` option
        v: true
      }
    };
  },
  Group({ node }) {
    if (!node.flags) {
      return;
    }
    const { enable, disable } = node.flags;
    enable?.extended && delete enable.extended;
    disable?.extended && delete disable.extended;
    enable?.dotAll && disable?.dotAll && delete enable.dotAll;
    enable?.ignoreCase && disable?.ignoreCase && delete enable.ignoreCase;
    enable && !Object.keys(enable).length && delete node.flags.enable;
    disable && !Object.keys(disable).length && delete node.flags.disable;
    !node.flags.enable && !node.flags.disable && delete node.flags;
  },
  Pattern({ node }, { accuracy, supportedGNodes }) {
    const leadingGs = [];
    let hasAltWithLeadG = false;
    let hasAltWithoutLeadG = false;
    for (const alt of node.alternatives) {
      const leadingG = getLeadingG(alt.elements);
      if (leadingG) {
        hasAltWithLeadG = true;
        if (Array.isArray(leadingG)) {
          leadingGs.push(...leadingG);
        } else {
          leadingGs.push(leadingG);
        }
      } else {
        hasAltWithoutLeadG = true;
      }
    }
    if (hasAltWithLeadG && hasAltWithoutLeadG && accuracy !== "loose") {
      throw new Error(r2`Uses "\G" in a way that's unsupported`);
    }
    leadingGs.forEach((g) => supportedGNodes.add(g));
  },
  Quantifier({ node }) {
    if (node.element.type === AstTypes.Quantifier) {
      const group = prepContainer(createGroup(), [node.element]);
      group.parent = node;
      node.element = group;
    }
  },
  VariableLengthCharacterSet({ node, replaceWith }, { accuracy, minTargetEs2024 }) {
    const { kind } = node;
    if (kind === AstVariableLengthCharacterSetKinds.newline) {
      replaceWith(parseFragment("(?>\r\n?|[\n\v\f\x85\u2028\u2029])"));
    } else if (kind === AstVariableLengthCharacterSetKinds.grapheme) {
      if (accuracy === "strict") {
        throw new Error(r2`Use of "\X" requires non-strict accuracy`);
      }
      const emoji = minTargetEs2024 ? r2`\p{RGI_Emoji}` : emoji_regex_xs_default().source.replace(/\\u\{/g, r2`\x{`);
      replaceWith(parseFragment(r2`(?>\r\n|${emoji}|\P{M}\p{M}*)`, { skipPropertyNameValidation: true }));
    } else {
      throw new Error(`Unexpected varcharset kind "${kind}"`);
    }
  }
};
var SecondPassVisitor = {
  Alternative({ node }, { namedGroupsInScopeByAlt }) {
    const parentAlt = getParentAlternative(node);
    if (parentAlt) {
      const groups = namedGroupsInScopeByAlt.get(parentAlt);
      if (groups) {
        namedGroupsInScopeByAlt.set(node, groups);
      }
    }
  },
  Backreference({ node }, { multiplexCapturesToLeftByRef, reffedNodesByBackreference }) {
    const { orphan, ref } = node;
    if (!orphan) {
      reffedNodesByBackreference.set(node, [...multiplexCapturesToLeftByRef.get(ref).map(({ node: node2 }) => node2)]);
    }
  },
  CapturingGroup: {
    enter({
      node,
      replaceWith,
      skip
    }, {
      groupOriginByCopy,
      groupsWithDuplicateNamesToRemove,
      multiplexCapturesToLeftByRef,
      namedGroupsInScopeByAlt,
      openDirectCaptures,
      openSubroutineRefs
    }) {
      const { name, number } = node;
      const ref = name ?? number;
      const origin = groupOriginByCopy.get(node);
      const parentAlt = getParentAlternative(node);
      if (openSubroutineRefs.has(ref)) {
        throw new Error("Unsupported indirect recursion");
      }
      if (origin) {
        openSubroutineRefs.add(ref);
      } else {
        openDirectCaptures.add(node);
      }
      if (openDirectCaptures.has(origin)) {
        replaceWith(createRecursion(ref));
        skip();
        return;
      }
      const multiplexNodes = getOrCreate(multiplexCapturesToLeftByRef, ref, []);
      for (let i = 0; i < multiplexNodes.length; i++) {
        const multiplex = multiplexNodes[i];
        if (
          // This group is from subroutine expansion, and there's a multiplex value from either the
          // origin node or a prior subroutine expansion group with the same origin
          origin === multiplex.node || origin && origin === multiplex.origin || // This group is not from subroutine expansion, and it comes after a subroutine expansion
          // group that refers to this group
          node === multiplex.origin
        ) {
          multiplexNodes.splice(i, 1);
          break;
        }
      }
      multiplexNodes.push({ node, origin });
      if (name) {
        const namedGroupsInScope = getOrCreate(namedGroupsInScopeByAlt, parentAlt, /* @__PURE__ */ new Map());
        if (namedGroupsInScope.has(name)) {
          groupsWithDuplicateNamesToRemove.add(namedGroupsInScope.get(name));
        }
        namedGroupsInScope.set(name, node);
        let upAlt = getParentAlternative(parentAlt);
        if (upAlt) {
          do {
            getOrCreate(namedGroupsInScopeByAlt, upAlt, /* @__PURE__ */ new Map()).set(name, node);
          } while (upAlt = getParentAlternative(upAlt));
        }
      }
    },
    exit({ node }, { groupOriginByCopy, openDirectCaptures, openSubroutineRefs }) {
      const { name, number } = node;
      if (groupOriginByCopy.get(node)) {
        openSubroutineRefs.delete(name ?? number);
      } else {
        openDirectCaptures.delete(node);
      }
    }
  },
  Group: {
    enter({ node }, state) {
      state.prevFlags = state.currentFlags;
      if (node.flags) {
        state.currentFlags = getNewCurrentFlags(state.currentFlags, node.flags);
      }
    },
    exit(_, state) {
      state.currentFlags = state.prevFlags;
    }
  },
  Subroutine(path, state) {
    const { node, replaceWith } = path;
    const { ref } = node;
    const reffedGroupNode = state.subroutineRefMap.get(ref);
    const isGlobalRecursion = ref === 0;
    const expandedSubroutine = isGlobalRecursion ? createRecursion(ref) : (
      // The reffed group might itself contain subroutines, which are expanded during sub-traversal
      cloneCapturingGroup(reffedGroupNode, state.groupOriginByCopy, null)
    );
    let replacement = expandedSubroutine;
    if (!isGlobalRecursion) {
      const reffedGroupFlagMods = getCombinedFlagModsFromFlagNodes(getAllParents(reffedGroupNode, (node2) => {
        return node2.type === AstTypes.Group && !!node2.flags;
      }));
      const reffedGroupFlags = reffedGroupFlagMods ? getNewCurrentFlags(state.globalFlags, reffedGroupFlagMods) : state.globalFlags;
      if (!areFlagsEqual(reffedGroupFlags, state.currentFlags)) {
        replacement = prepContainer(createGroup({
          flags: getFlagModsFromFlags(reffedGroupFlags)
        }), [expandedSubroutine]);
      }
    }
    replaceWith(replacement);
    if (!isGlobalRecursion) {
      traverseReplacement(replacement, path, state, SecondPassVisitor);
    }
  }
};
var ThirdPassVisitor = {
  Backreference({ node, replaceWith }, state) {
    if (node.orphan) {
      state.highestOrphanBackref = Math.max(state.highestOrphanBackref, node.ref);
      return;
    }
    const reffedNodes = state.reffedNodesByBackreference.get(node);
    const participants = reffedNodes.filter((reffedNode) => canCaptureParticipateForBackref(reffedNode, node));
    if (!participants.length) {
      replaceWith(createLookaround({ negate: true }));
    } else if (participants.length > 1) {
      const alts = participants.map((reffedNode) => adoptAndSwapKids(
        createAlternative(),
        [createBackreference(reffedNode.number)]
      ));
      replaceWith(adoptAndSwapKids(createGroup(), alts));
    } else {
      node.ref = participants[0].number;
    }
  },
  CapturingGroup({ node }, state) {
    node.number = ++state.numCapturesToLeft;
    if (state.groupsWithDuplicateNamesToRemove.has(node)) {
      delete node.name;
    }
  },
  Regex: {
    exit({ node }, state) {
      const numCapsNeeded = Math.max(state.highestOrphanBackref - state.numCapturesToLeft, 0);
      for (let i = 0; i < numCapsNeeded; i++) {
        const emptyCapture = createCapturingGroup();
        node.pattern.alternatives.at(-1).elements.push(emptyCapture);
      }
    }
  }
};
function adoptAndSwapKids(parent, kids) {
  kids.forEach((kid) => kid.parent = parent);
  parent[getContainerAccessor(parent)] = kids;
  return parent;
}
function areFlagsEqual(a, b) {
  return a.dotAll === b.dotAll && a.ignoreCase === b.ignoreCase;
}
function canCaptureParticipateForBackref(capture, backref) {
  let rightmostPoint = backref;
  do {
    if (rightmostPoint.type === AstTypes.Pattern) {
      return false;
    }
    if (rightmostPoint.type === AstTypes.Alternative) {
      continue;
    }
    if (rightmostPoint === capture) {
      return false;
    }
    const kidsOfParent = getKids(rightmostPoint.parent);
    for (const kid of kidsOfParent) {
      if (kid === rightmostPoint) {
        break;
      }
      if (kid === capture) {
        return true;
      }
      if (hasDescendant(kid, capture)) {
        return true;
      }
    }
  } while (rightmostPoint = rightmostPoint.parent);
  throw new Error("Unexpected path");
}
function cloneCapturingGroup(obj, originMap, up, up2) {
  const store = Array.isArray(obj) ? [] : {};
  for (const [key, value] of Object.entries(obj)) {
    if (key === "parent") {
      store.parent = Array.isArray(up) ? up2 : up;
    } else if (value && typeof value === "object") {
      store[key] = cloneCapturingGroup(value, originMap, store, up);
    } else {
      if (key === "type" && value === AstTypes.CapturingGroup) {
        originMap.set(store, originMap.get(obj) ?? obj);
      }
      store[key] = value;
    }
  }
  return store;
}
function createRecursion(ref) {
  return {
    type: AstTypes.Recursion,
    ref
  };
}
function getAllParents(node, filterFn) {
  const results = [];
  while (node = node.parent) {
    if (!filterFn || filterFn(node)) {
      results.push(node);
    }
  }
  return results;
}
function getContainerAccessor(node) {
  for (const accessor of ["alternatives", "classes", "elements"]) {
    if (node[accessor]) {
      return accessor;
    }
  }
  return null;
}
function getCombinedFlagModsFromFlagNodes(flagNodes) {
  const flagProps = ["dotAll", "ignoreCase"];
  const combinedFlags = { enable: {}, disable: {} };
  flagNodes.forEach(({ flags }) => {
    flagProps.forEach((prop) => {
      if (flags.enable?.[prop]) {
        delete combinedFlags.disable[prop];
        combinedFlags.enable[prop] = true;
      }
      if (flags.disable?.[prop]) {
        combinedFlags.disable[prop] = true;
      }
    });
  });
  if (!Object.keys(combinedFlags.enable).length) {
    delete combinedFlags.enable;
  }
  if (!Object.keys(combinedFlags.disable).length) {
    delete combinedFlags.disable;
  }
  if (combinedFlags.enable || combinedFlags.disable) {
    return combinedFlags;
  }
  return null;
}
function getFlagModsFromFlags({ dotAll, ignoreCase }) {
  const mods = {};
  if (dotAll || ignoreCase) {
    mods.enable = {};
    dotAll && (mods.enable.dotAll = true);
    ignoreCase && (mods.enable.ignoreCase = true);
  }
  if (!dotAll || !ignoreCase) {
    mods.disable = {};
    !dotAll && (mods.disable.dotAll = true);
    !ignoreCase && (mods.disable.ignoreCase = true);
  }
  return mods;
}
function getKids(node) {
  if (!node) {
    throw new Error("Node expected");
  }
  if (node.type === AstTypes.Quantifier) {
    return [node.element];
  }
  const accessor = getContainerAccessor(node);
  return accessor && node[accessor];
}
function getLeadingG(els) {
  if (!els.length) {
    return null;
  }
  const first = els[0];
  if (isLookaround(first) && !first.negate && first.alternatives.length === 1 && first.alternatives[0].elements.length) {
    const els2 = first.alternatives[0].elements;
    const index = first.kind === AstAssertionKinds.lookahead ? 0 : els2.length - 1;
    if (els2[index].kind === AstAssertionKinds.search_start) {
      return els2[index];
    }
  }
  const firstToConsider = els.find((el) => {
    return el.kind === AstAssertionKinds.search_start ? true : el.type !== AstTypes.Directive && el.type !== AstTypes.Assertion && !(el.type === AstTypes.Quantifier && !el.min);
  });
  if (!firstToConsider) {
    return null;
  }
  if (firstToConsider.kind === AstAssertionKinds.search_start) {
    return firstToConsider;
  }
  if (firstToConsider.type === AstTypes.Group || firstToConsider.type === AstTypes.CapturingGroup) {
    const gNodesForGroup = [];
    for (const alt of firstToConsider.alternatives) {
      const leadingG = getLeadingG(alt.elements);
      if (!leadingG) {
        return null;
      }
      if (Array.isArray(leadingG)) {
        gNodesForGroup.push(...leadingG);
      } else {
        gNodesForGroup.push(leadingG);
      }
    }
    return gNodesForGroup;
  }
  return null;
}
function getParentAlternative(node) {
  while (node = node.parent) {
    if (node.type === AstTypes.Alternative) {
      return node;
    }
  }
  return null;
}
function hasDescendant(node, descendant) {
  const kids = getKids(node) ?? [];
  for (const kid of kids) {
    if (kid === descendant || hasDescendant(kid, descendant)) {
      return true;
    }
  }
  return false;
}
function isValidGroupNameJs(name) {
  return /^[$_\p{IDS}][$\u200C\u200D\p{IDC}]*$/u.test(name);
}
function parseFragment(pattern2, { skipPropertyNameValidation } = {}) {
  const ast = parse(tokenize(pattern2), { skipPropertyNameValidation });
  const alts = ast.pattern.alternatives;
  if (alts.length > 1 || alts[0].elements.length > 1) {
    return adoptAndSwapKids(createGroup(), alts);
  }
  return alts[0].elements[0];
}
function prepContainer(node, kids) {
  const accessor = getContainerAccessor(node);
  node[accessor][0].parent = node;
  if (kids) {
    adoptAndSwapKids(node[accessor][0], kids);
  }
  return node;
}
function traverseReplacement(replacement, { parent, key, container }, state, visitor) {
  traverse({
    // Don't use the `node` from `path`
    node: replacement,
    parent,
    key,
    container
  }, state, visitor);
}

// src/generate.js
function generate(ast, options) {
  const opts = getOptions(options);
  const minTargetEs2024 = isMinTarget(opts.target, "ES2024");
  const minTargetEs2025 = isMinTarget(opts.target, "ES2025");
  const rDepth = opts.maxRecursionDepth;
  if (rDepth !== null && (!Number.isInteger(rDepth) || rDepth < 2 || rDepth > 100)) {
    throw new Error("Invalid maxRecursionDepth; use 2-100 or null");
  }
  let hasCaseInsensitiveNode = null;
  let hasCaseSensitiveNode = null;
  if (!minTargetEs2025) {
    const iStack = [ast.flags.ignoreCase];
    traverse({ node: ast }, {
      getCurrentModI: () => iStack.at(-1),
      popModI() {
        iStack.pop();
      },
      pushModI(isIOn) {
        iStack.push(isIOn);
      },
      setHasCasedChar() {
        if (iStack.at(-1)) {
          hasCaseInsensitiveNode = true;
        } else {
          hasCaseSensitiveNode = true;
        }
      }
    }, FlagModifierVisitor);
  }
  const appliedGlobalFlags = {
    dotAll: ast.flags.dotAll,
    // - Turn global flag i on if a case insensitive node was used and no case sensitive nodes were
    //   used (to avoid unnecessary node expansion).
    // - Turn global flag i off if a case sensitive node was used (since case sensitivity can't be
    //   forced without the use of ES2025 flag groups)
    ignoreCase: !!((ast.flags.ignoreCase || hasCaseInsensitiveNode) && !hasCaseSensitiveNode)
  };
  let lastNode = null;
  const state = {
    accuracy: opts.accuracy,
    appliedGlobalFlags,
    captureFlagIMap: /* @__PURE__ */ new Map(),
    currentFlags: {
      dotAll: ast.flags.dotAll,
      ignoreCase: ast.flags.ignoreCase
    },
    groupNames: /* @__PURE__ */ new Set(),
    inCharClass: false,
    lastNode,
    maxRecursionDepth: rDepth,
    useAppliedIgnoreCase: !!(!minTargetEs2025 && hasCaseInsensitiveNode && hasCaseSensitiveNode),
    useDuplicateNames: minTargetEs2025,
    useFlagMods: minTargetEs2025,
    useFlagV: minTargetEs2024,
    usePostEs2018Properties: minTargetEs2024,
    verbose: opts.verbose
  };
  function gen(node) {
    state.lastNode = lastNode;
    lastNode = node;
    switch (node.type) {
      case AstTypes.Regex:
        return {
          pattern: gen(node.pattern),
          flags: gen(node.flags),
          options: { ...node.options }
        };
      case AstTypes.Alternative:
        return node.elements.map(gen).join("");
      case AstTypes.Assertion:
        return genAssertion(node, state, gen);
      case AstTypes.Backreference:
        return genBackreference(node, state);
      case AstTypes.CapturingGroup:
        return genCapturingGroup(node, state, gen);
      case AstTypes.Character:
        return genCharacter(node, state);
      case AstTypes.CharacterClass:
        return genCharacterClass(node, state, gen);
      case AstTypes.CharacterClassIntersection:
        if (!state.useFlagV) {
          throw new Error("Use of class intersection requires min target ES2024");
        }
        return node.classes.map(gen).join("&&");
      case AstTypes.CharacterClassRange:
        return genCharacterClassRange(node, state);
      case AstTypes.CharacterSet:
        return genCharacterSet(node, state);
      case AstTypes.Flags:
        return genFlags(node, state);
      case AstTypes.Group:
        return genGroup(node, state, gen);
      case AstTypes.Pattern:
        return node.alternatives.map(gen).join("|");
      case AstTypes.Quantifier:
        return gen(node.element) + getQuantifierStr(node);
      case AstTypes.Recursion:
        return genRecursion(node, state);
      default:
        throw new Error(`Unexpected node type "${node.type}"`);
    }
  }
  const result = gen(ast);
  if (!minTargetEs2024) {
    delete result.options.force.v;
    result.options.disable.v = true;
    result.options.unicodeSetsPlugin = null;
  }
  return result;
}
var FlagModifierVisitor = {
  AnyGroup: {
    enter({ node }, state) {
      const currentModI = state.getCurrentModI();
      state.pushModI(
        node.flags ? getNewCurrentFlags({ ignoreCase: currentModI }, node.flags).ignoreCase : currentModI
      );
    },
    exit(_, state) {
      state.popModI();
    }
  },
  Backreference(_, state) {
    state.setHasCasedChar();
  },
  Character({ node }, state) {
    if (charHasCase(cp(node.value))) {
      state.setHasCasedChar();
    }
  },
  CharacterClassRange({ node, skip }, state) {
    skip();
    if (getCasesOutsideCharClassRange(node, { firstOnly: true }).length) {
      state.setHasCasedChar();
    }
  },
  CharacterSet({ node }, state) {
    if (node.kind === AstCharacterSetKinds.property && UnicodePropertiesWithSpecificCase.has(node.value)) {
      state.setHasCasedChar();
    }
  }
};
var BaseEscapeChars = /* @__PURE__ */ new Set([
  "$",
  "(",
  ")",
  "*",
  "+",
  ".",
  "?",
  "[",
  "\\",
  "]",
  "^",
  "{",
  "|",
  "}"
]);
var CharClassEscapeChars = /* @__PURE__ */ new Set([
  "-",
  "\\",
  "]",
  "^"
]);
var CharClassEscapeCharsFlagV = /* @__PURE__ */ new Set([
  "(",
  ")",
  "-",
  "/",
  "[",
  "\\",
  "]",
  "^",
  "{",
  "|",
  "}",
  // Double punctuators; also includes already-listed `-` and `^`
  "!",
  "#",
  "$",
  "%",
  "&",
  "*",
  "+",
  ",",
  ".",
  ":",
  ";",
  "<",
  "=",
  ">",
  "?",
  "@",
  "`",
  "~"
]);
var CharCodeEscapeMap = /* @__PURE__ */ new Map([
  [9, r2`\t`],
  // horizontal tab
  [10, r2`\n`],
  // line feed
  [11, r2`\v`],
  // vertical tab
  [12, r2`\f`],
  // form feed
  [13, r2`\r`],
  // carriage return
  [8232, r2`\u2028`],
  // line separator
  [8233, r2`\u2029`],
  // paragraph separator
  [65279, r2`\uFEFF`]
  // ZWNBSP/BOM
]);
var casedRe = /^\p{Cased}$/u;
function charHasCase(char) {
  return casedRe.test(char);
}
function genAssertion(node, _, gen) {
  const { kind, negate, alternatives } = node;
  if (isLookaround(node)) {
    const prefix = `${kind === AstAssertionKinds.lookahead ? "" : "<"}${negate ? "!" : "="}`;
    return `(?${prefix}${alternatives.map(gen).join("|")})`;
  }
  if (kind === AstAssertionKinds.string_end) {
    return "$";
  }
  if (kind === AstAssertionKinds.string_start) {
    return "^";
  }
  throw new Error(`Unexpected assertion kind "${kind}"`);
}
function genBackreference({ ref }, state) {
  if (typeof ref !== "number") {
    throw new Error("Unexpected named backref in transformed AST");
  }
  if (!state.useFlagMods && state.accuracy === "strict" && state.currentFlags.ignoreCase && !state.captureFlagIMap.get(ref)) {
    throw new Error("Use of case-insensitive backref to case-sensitive group requires target ES2025 or non-strict accuracy");
  }
  return "\\" + ref;
}
function genCapturingGroup({ name, number, alternatives }, state, gen) {
  if (name) {
    if (state.groupNames.has(name)) {
      if (!state.useDuplicateNames) {
        name = null;
      }
    } else {
      state.groupNames.add(name);
    }
  }
  state.captureFlagIMap.set(number, state.currentFlags.ignoreCase);
  return `(${name ? `?<${name}>` : ""}${alternatives.map(gen).join("|")})`;
}
function genCharacter({ value }, state) {
  const char = cp(value);
  const escaped = getCharEscape(value, {
    isAfterBackref: state.lastNode.type === AstTypes.Backreference,
    inCharClass: state.inCharClass,
    useFlagV: state.useFlagV
  });
  if (escaped !== char) {
    return escaped;
  }
  if (state.useAppliedIgnoreCase && state.currentFlags.ignoreCase && charHasCase(char)) {
    const cases = getIgnoreCaseMatchChars(char);
    return state.inCharClass ? cases.join("") : cases.length > 1 ? `[${cases.join("")}]` : cases[0];
  }
  return char;
}
function genCharacterClass({ negate, parent, elements }, state, gen) {
  if (!negate && // Allows many nested classes to work with `target` ES2018 which doesn't support nesting
  ((!state.useFlagV || !state.verbose) && parent.type === AstTypes.CharacterClass && elements[0].type !== AstTypes.CharacterClassIntersection) || !state.verbose && parent.type === AstTypes.CharacterClassIntersection && elements.length === 1 && elements[0].type !== AstTypes.CharacterClassRange) {
    return elements.map(gen).join("");
  }
  if (!state.useFlagV && parent.type === AstTypes.CharacterClass) {
    throw new Error("Use of nested character class requires min target ES2024");
  }
  state.inCharClass = true;
  const result = `[${negate ? "^" : ""}${elements.map(gen).join("")}]`;
  state.inCharClass = false;
  return result;
}
function genCharacterClassRange(node, state) {
  const min = node.min.value;
  const max = node.max.value;
  const escOpts = {
    isAfterBackref: false,
    inCharClass: true,
    useFlagV: state.useFlagV
  };
  const minStr = getCharEscape(min, escOpts);
  const maxStr = getCharEscape(max, escOpts);
  let extraChars = "";
  if (state.useAppliedIgnoreCase && state.currentFlags.ignoreCase) {
    const charsOutsideRange = getCasesOutsideCharClassRange(node);
    const ranges = getCodePointRangesFromChars(charsOutsideRange);
    ranges.forEach((value) => {
      extraChars += Array.isArray(value) ? `${getCharEscape(value[0], escOpts)}-${getCharEscape(value[1], escOpts)}` : getCharEscape(value, escOpts);
    });
  }
  return `${minStr}-${maxStr}${extraChars}`;
}
function genCharacterSet({ kind, negate, value, key }, state) {
  if (kind === AstCharacterSetKinds.dot) {
    return state.currentFlags.dotAll ? state.appliedGlobalFlags.dotAll || state.useFlagMods ? "." : "[^]" : (
      // Onig's only line break char is line feed, unlike JS
      r2`[^\n]`
    );
  }
  if (kind === AstCharacterSetKinds.digit) {
    return negate ? r2`\D` : r2`\d`;
  }
  if (kind === AstCharacterSetKinds.property) {
    if (!state.usePostEs2018Properties && JsUnicodePropertiesPostEs2018.has(value)) {
      throw new Error(`Unicode property "${value}" unavailable in target ES2018`);
    }
    if (state.useAppliedIgnoreCase && state.currentFlags.ignoreCase && UnicodePropertiesWithSpecificCase.has(value)) {
      throw new Error(`Unicode property "${value}" can't be case-insensitive when other chars have specific case`);
    }
    return `${negate ? r2`\P` : r2`\p`}{${key ? `${key}=` : ""}${value}}`;
  }
  if (kind === AstCharacterSetKinds.word) {
    return negate ? r2`\W` : r2`\w`;
  }
  throw new Error(`Unexpected character set kind "${kind}"`);
}
function genFlags(node, state) {
  return (
    // The transformer should never turn on the properties for flags d, g, and m since Onig doesn't
    // have equivs. Flag m is never relied on since Onig uses different line break chars than JS
    // (node.hasIndices ? 'd' : '') +
    // (node.global ? 'g' : '') +
    // (node.multiline ? 'm' : '') +
    (state.appliedGlobalFlags.ignoreCase ? "i" : "") + (node.dotAll ? "s" : "") + (node.sticky ? "y" : "")
  );
}
function genGroup({ atomic: atomic2, flags, parent, alternatives }, state, gen) {
  const currentFlags = state.currentFlags;
  if (flags) {
    state.currentFlags = getNewCurrentFlags(currentFlags, flags);
  }
  const contents = alternatives.map(gen).join("|");
  const result = !state.verbose && alternatives.length === 1 && parent.type !== AstTypes.Quantifier && !atomic2 && (!state.useFlagMods || !flags) ? contents : `(?${getGroupPrefix(atomic2, flags, state.useFlagMods)}${contents})`;
  state.currentFlags = currentFlags;
  return result;
}
function genRecursion({ ref }, state) {
  const rDepth = state.maxRecursionDepth;
  if (!rDepth) {
    throw new Error("Use of recursion disabled");
  }
  if (state.accuracy === "strict") {
    throw new Error("Use of recursion requires non-strict accuracy due to depth limit");
  }
  return ref === 0 ? `(?R=${rDepth})` : r2`\g<${ref}&R=${rDepth}>`;
}
function getCasesOutsideCharClassRange(node, { firstOnly } = {}) {
  const min = node.min.value;
  const max = node.max.value;
  const found = [];
  if (min < 65 && (max === 65535 || max >= 131071) || min === 65536 && max >= 131071) {
    return found;
  }
  for (let i = min; i <= max; i++) {
    const char = cp(i);
    if (!charHasCase(char)) {
      continue;
    }
    const charsOutsideRange = getIgnoreCaseMatchChars(char).filter((caseOfChar) => {
      const num = caseOfChar.codePointAt(0);
      return num < min || num > max;
    });
    if (charsOutsideRange.length) {
      found.push(...charsOutsideRange);
      if (firstOnly) {
        break;
      }
    }
  }
  return found;
}
function getCharEscape(codePoint, { isAfterBackref, inCharClass, useFlagV }) {
  if (CharCodeEscapeMap.has(codePoint)) {
    return CharCodeEscapeMap.get(codePoint);
  }
  if (
    // Control chars, etc.; condition modeled on the Chrome developer console's display for strings
    codePoint < 32 || codePoint > 126 && codePoint < 160 || // Unicode planes 4-16; unassigned, special purpose, and private use area
    codePoint > 262143 || // Avoid corrupting a preceding backref by immediately following it with a literal digit
    isAfterBackref && isDigitCharCode(codePoint)
  ) {
    return codePoint > 255 ? r2`\u{${codePoint.toString(16).toUpperCase()}}` : r2`\x${codePoint.toString(16).toUpperCase().padStart(2, "0")}`;
  }
  const escapeChars = inCharClass ? useFlagV ? CharClassEscapeCharsFlagV : CharClassEscapeChars : BaseEscapeChars;
  const char = cp(codePoint);
  return (escapeChars.has(char) ? "\\" : "") + char;
}
function getCodePointRangesFromChars(chars) {
  const codePoints = chars.map((char) => char.codePointAt(0)).sort((a, b) => a - b);
  const values = [];
  let start = null;
  for (let i = 0; i < codePoints.length; i++) {
    if (codePoints[i + 1] === codePoints[i] + 1) {
      start ??= codePoints[i];
    } else if (start === null) {
      values.push(codePoints[i]);
    } else {
      values.push([start, codePoints[i]]);
      start = null;
    }
  }
  return values;
}
function getGroupPrefix(atomic2, flagMods, useFlagMods) {
  if (atomic2) {
    return ">";
  }
  let mods = "";
  if (flagMods && useFlagMods) {
    const { enable, disable } = flagMods;
    mods = (enable?.ignoreCase ? "i" : "") + (enable?.dotAll ? "s" : "") + (disable ? "-" : "") + (disable?.ignoreCase ? "i" : "") + (disable?.dotAll ? "s" : "");
  }
  return `${mods}:`;
}
function getQuantifierStr({ min, max, greedy, possessive: possessive2 }) {
  let base;
  if (!min && max === 1) {
    base = "?";
  } else if (!min && max === Infinity) {
    base = "*";
  } else if (min === 1 && max === Infinity) {
    base = "+";
  } else if (min === max) {
    base = `{${min}}`;
  } else {
    base = `{${min},${max === Infinity ? "" : max}}`;
  }
  return base + (possessive2 ? "+" : greedy ? "" : "?");
}
function isDigitCharCode(value) {
  return value > 47 && value < 58;
}

// node_modules/regex-utilities/src/index.js
var Context = Object.freeze({
  DEFAULT: "DEFAULT",
  CHAR_CLASS: "CHAR_CLASS"
});
function replaceUnescaped(expression, needle, replacement, context) {
  const re = new RegExp(String.raw`${needle}|(?<$skip>\[\^?|\\?.)`, "gsu");
  const negated = [false];
  let numCharClassesOpen = 0;
  let result = "";
  for (const match of expression.matchAll(re)) {
    const { 0: m, groups: { $skip } } = match;
    if (!$skip && (!context || context === Context.DEFAULT === !numCharClassesOpen)) {
      if (replacement instanceof Function) {
        result += replacement(match, {
          context: numCharClassesOpen ? Context.CHAR_CLASS : Context.DEFAULT,
          negated: negated[negated.length - 1]
        });
      } else {
        result += replacement;
      }
      continue;
    }
    if (m[0] === "[") {
      numCharClassesOpen++;
      negated.push(m[1] === "^");
    } else if (m === "]" && numCharClassesOpen) {
      numCharClassesOpen--;
      negated.pop();
    }
    result += m;
  }
  return result;
}
function forEachUnescaped(expression, needle, callback, context) {
  replaceUnescaped(expression, needle, callback, context);
}
function execUnescaped(expression, needle, pos = 0, context) {
  if (!new RegExp(needle, "su").test(expression)) {
    return null;
  }
  const re = new RegExp(`${needle}|(?<$skip>\\\\?.)`, "gsu");
  re.lastIndex = pos;
  let numCharClassesOpen = 0;
  let match;
  while (match = re.exec(expression)) {
    const { 0: m, groups: { $skip } } = match;
    if (!$skip && (!context || context === Context.DEFAULT === !numCharClassesOpen)) {
      return match;
    }
    if (m === "[") {
      numCharClassesOpen++;
    } else if (m === "]" && numCharClassesOpen) {
      numCharClassesOpen--;
    }
    if (re.lastIndex == match.index) {
      re.lastIndex++;
    }
  }
  return null;
}
function hasUnescaped(expression, needle, context) {
  return !!execUnescaped(expression, needle, 0, context);
}
function getGroupContents(expression, contentsStartPos) {
  const token2 = /\\?./gsu;
  token2.lastIndex = contentsStartPos;
  let contentsEndPos = expression.length;
  let numCharClassesOpen = 0;
  let numGroupsOpen = 1;
  let match;
  while (match = token2.exec(expression)) {
    const [m] = match;
    if (m === "[") {
      numCharClassesOpen++;
    } else if (!numCharClassesOpen) {
      if (m === "(") {
        numGroupsOpen++;
      } else if (m === ")") {
        numGroupsOpen--;
        if (!numGroupsOpen) {
          contentsEndPos = match.index;
          break;
        }
      }
    } else if (m === "]") {
      numCharClassesOpen--;
    }
  }
  return expression.slice(contentsStartPos, contentsEndPos);
}

// node_modules/regex/src/utils.js
var RegexContext = {
  DEFAULT: "DEFAULT",
  CHAR_CLASS: "CHAR_CLASS",
  ENCLOSED_P: "ENCLOSED_P",
  ENCLOSED_U: "ENCLOSED_U",
  GROUP_NAME: "GROUP_NAME",
  INTERVAL_QUANTIFIER: "INTERVAL_QUANTIFIER",
  INVALID_INCOMPLETE_TOKEN: "INVALID_INCOMPLETE_TOKEN"
};
var CharClassContext = {
  DEFAULT: "DEFAULT",
  ENCLOSED_P: "ENCLOSED_P",
  ENCLOSED_Q: "ENCLOSED_Q",
  ENCLOSED_U: "ENCLOSED_U",
  INVALID_INCOMPLETE_TOKEN: "INVALID_INCOMPLETE_TOKEN",
  RANGE: "RANGE"
};
var enclosedTokenRegexContexts = /* @__PURE__ */ new Set([
  RegexContext.ENCLOSED_P,
  RegexContext.ENCLOSED_U
]);
var enclosedTokenCharClassContexts = /* @__PURE__ */ new Set([
  CharClassContext.ENCLOSED_P,
  CharClassContext.ENCLOSED_Q,
  CharClassContext.ENCLOSED_U
]);
var patternModsSupported = (() => {
  try {
    new RegExp("(?i:)");
  } catch (e) {
    return false;
  }
  return true;
})();
var flagVSupported = (() => {
  try {
    new RegExp("", "v");
  } catch (e) {
    return false;
  }
  return true;
})();
var emulationGroupMarker = "$E$";
var namedCapturingDelim = String.raw`\(\?<(?![=!])(?<captureName>[^>]+)>`;
var capturingDelim = String.raw`\((?!\?)(?!(?<=\(\?\()DEFINE\))|${namedCapturingDelim}`;
var noncapturingDelim = String.raw`\(\?(?:[:=!>A-Za-z\-]|<[=!]|\(DEFINE\))`;
var stringPropertyNames = [
  "Basic_Emoji",
  "Emoji_Keycap_Sequence",
  "RGI_Emoji_Modifier_Sequence",
  "RGI_Emoji_Flag_Sequence",
  "RGI_Emoji_Tag_Sequence",
  "RGI_Emoji_ZWJ_Sequence",
  "RGI_Emoji"
].join("|");
var charClassUnionToken = new RegExp(String.raw`
\\(?: c[A-Za-z]
  | p\{(?<pStrProp>${stringPropertyNames})\}
  | [pP]\{[^\}]+\}
  | (?<qStrProp>q)
  | u(?:[A-Fa-f\d]{4}|\{[A-Fa-f\d]+\})
  | x[A-Fa-f\d]{2}
  | .
)
| --
| &&
| .
`.replace(/\s+/g, ""), "gsu");
var contextToken = new RegExp(String.raw`
(?<groupN>\(\?<(?![=!])|\\[gk]<)
| (?<enclosedPU>\\[pPu]\{)
| (?<enclosedQ>\\q\{)
| (?<intervalQ>\{)
| (?<incompleteT>\\(?: $
  | c(?![A-Za-z])
  | u(?![A-Fa-f\d]{4})[A-Fa-f\d]{0,3}
  | x(?![A-Fa-f\d]{2})[A-Fa-f\d]?
  )
)
| --
| \\?.
`.replace(/\s+/g, ""), "gsu");
function spliceStr(str, pos, oldValue, newValue) {
  return str.slice(0, pos) + newValue + str.slice(pos + oldValue.length);
}

// node_modules/regex/src/atomic.js
var atomicPluginToken = new RegExp(String.raw`(?<noncapturingStart>${noncapturingDelim})|(?<capturingStart>\((?:\?<[^>]+>)?)|\\?.`, "gsu");
function atomic(expression, data) {
  if (!/\(\?>/.test(expression)) {
    return expression;
  }
  const aGDelim = "(?>";
  const emulatedAGDelim = `(?:(?=(${data?.useEmulationGroups ? emulationGroupMarker : ""}`;
  const captureNumMap = [0];
  let numCapturesBeforeAG = 0;
  let numAGs = 0;
  let aGPos = NaN;
  let hasProcessedAG;
  do {
    hasProcessedAG = false;
    let numCharClassesOpen = 0;
    let numGroupsOpenInAG = 0;
    let inAG = false;
    let match;
    atomicPluginToken.lastIndex = Number.isNaN(aGPos) ? 0 : aGPos + emulatedAGDelim.length;
    while (match = atomicPluginToken.exec(expression)) {
      const { 0: m, index, groups: { capturingStart, noncapturingStart } } = match;
      if (m === "[") {
        numCharClassesOpen++;
      } else if (!numCharClassesOpen) {
        if (m === aGDelim && !inAG) {
          aGPos = index;
          inAG = true;
        } else if (inAG && noncapturingStart) {
          numGroupsOpenInAG++;
        } else if (capturingStart) {
          if (inAG) {
            numGroupsOpenInAG++;
          } else {
            numCapturesBeforeAG++;
            captureNumMap.push(numCapturesBeforeAG + numAGs);
          }
        } else if (m === ")" && inAG) {
          if (!numGroupsOpenInAG) {
            numAGs++;
            expression = `${expression.slice(0, aGPos)}${emulatedAGDelim}${expression.slice(aGPos + aGDelim.length, index)}))<$$${numAGs + numCapturesBeforeAG}>)${expression.slice(index + 1)}`;
            hasProcessedAG = true;
            break;
          }
          numGroupsOpenInAG--;
        }
      } else if (m === "]") {
        numCharClassesOpen--;
      }
    }
  } while (hasProcessedAG);
  expression = replaceUnescaped(
    expression,
    String.raw`\\(?<backrefNum>[1-9]\d*)|<\$\$(?<wrappedBackrefNum>\d+)>`,
    ({ 0: m, groups: { backrefNum, wrappedBackrefNum } }) => {
      if (backrefNum) {
        const bNum = +backrefNum;
        if (bNum > captureNumMap.length - 1) {
          throw new Error(`Backref "${m}" greater than number of captures`);
        }
        return `\\${captureNumMap[bNum]}`;
      }
      return `\\${wrappedBackrefNum}`;
    },
    Context.DEFAULT
  );
  return expression;
}
var baseQuantifier = String.raw`(?:[?*+]|\{\d+(?:,\d*)?\})`;
var possessivePluginToken = new RegExp(String.raw`
\\(?: \d+
  | c[A-Za-z]
  | [gk]<[^>]+>
  | [pPu]\{[^\}]+\}
  | u[A-Fa-f\d]{4}
  | x[A-Fa-f\d]{2}
  )
| \((?: \? (?: [:=!>]
  | <(?:[=!]|[^>]+>)
  | [A-Za-z\-]+:
  | \(DEFINE\)
  ))?
| (?<qBase>${baseQuantifier})(?<qMod>[?+]?)(?<invalidQ>[?*+\{]?)
| \\?.
`.replace(/\s+/g, ""), "gsu");
function possessive(expression) {
  if (!new RegExp(`${baseQuantifier}\\+`).test(expression)) {
    return expression;
  }
  const openGroupIndices = [];
  let lastGroupIndex = null;
  let lastCharClassIndex = null;
  let lastToken = "";
  let numCharClassesOpen = 0;
  let match;
  possessivePluginToken.lastIndex = 0;
  while (match = possessivePluginToken.exec(expression)) {
    const { 0: m, index, groups: { qBase, qMod, invalidQ } } = match;
    if (m === "[") {
      if (!numCharClassesOpen) {
        lastCharClassIndex = index;
      }
      numCharClassesOpen++;
    } else if (m === "]") {
      if (numCharClassesOpen) {
        numCharClassesOpen--;
      } else {
        lastCharClassIndex = null;
      }
    } else if (!numCharClassesOpen) {
      if (qMod === "+" && lastToken && !lastToken.startsWith("(")) {
        if (invalidQ) {
          throw new Error(`Invalid quantifier "${m}"`);
        }
        let charsAdded = -1;
        if (/^\{\d+\}$/.test(qBase)) {
          expression = spliceStr(expression, index + qBase.length, qMod, "");
        } else {
          if (lastToken === ")" || lastToken === "]") {
            const nodeIndex = lastToken === ")" ? lastGroupIndex : lastCharClassIndex;
            if (nodeIndex === null) {
              throw new Error(`Invalid unmatched "${lastToken}"`);
            }
            expression = `${expression.slice(0, nodeIndex)}(?>${expression.slice(nodeIndex, index)}${qBase})${expression.slice(index + m.length)}`;
          } else {
            expression = `${expression.slice(0, index - lastToken.length)}(?>${lastToken}${qBase})${expression.slice(index + m.length)}`;
          }
          charsAdded += 4;
        }
        possessivePluginToken.lastIndex += charsAdded;
      } else if (m[0] === "(") {
        openGroupIndices.push(index);
      } else if (m === ")") {
        lastGroupIndex = openGroupIndices.length ? openGroupIndices.pop() : null;
      }
    }
    lastToken = m;
  }
  return expression;
}

// node_modules/regex-recursion/src/index.js
var gRToken = String.raw`\\g<(?<gRNameOrNum>[^>&]+)&R=(?<gRDepth>[^>]+)>`;
var recursiveToken = String.raw`\(\?R=(?<rDepth>[^\)]+)\)|${gRToken}`;
var namedCapturingDelim2 = String.raw`\(\?<(?![=!])(?<captureName>[^>]+)>`;
var token = new RegExp(String.raw`${namedCapturingDelim2}|${recursiveToken}|\\?.`, "gsu");
function recursion(expression) {
  if (!new RegExp(recursiveToken, "su").test(expression)) {
    return expression;
  }
  if (hasUnescaped(expression, String.raw`\\[1-9]`, Context.DEFAULT)) {
    throw new Error(`Numbered backrefs cannot be used with recursion; use named backref`);
  }
  if (hasUnescaped(expression, String.raw`\(\?\(DEFINE\)`, Context.DEFAULT)) {
    throw new Error(`DEFINE groups cannot be used with recursion`);
  }
  const groupContentsStartPos = /* @__PURE__ */ new Map();
  const openGroups = [];
  let numCharClassesOpen = 0;
  let numCaptures = 0;
  let match;
  token.lastIndex = 0;
  while (match = token.exec(expression)) {
    const { 0: m, groups: { captureName, rDepth, gRNameOrNum, gRDepth } } = match;
    if (m === "[") {
      numCharClassesOpen++;
    } else if (!numCharClassesOpen) {
      if (rDepth) {
        assertMaxInBounds(rDepth);
        const pre = expression.slice(0, match.index);
        const post = expression.slice(token.lastIndex);
        assertNoFollowingRecursion(post);
        return makeRecursive(pre, post, +rDepth, false);
      } else if (gRNameOrNum) {
        assertMaxInBounds(gRDepth);
        assertNoFollowingRecursion(expression.slice(token.lastIndex));
        if (!openGroups.some((g) => g.name === gRNameOrNum || g.num === +gRNameOrNum)) {
          throw new Error(`Recursion via \\g<${gRNameOrNum}&R=${gRDepth}> must be used within the referenced group`);
        }
        const startPos = groupContentsStartPos.get(gRNameOrNum);
        const recursiveGroupContents = getGroupContents(expression, startPos);
        const pre = expression.slice(startPos, match.index);
        const post = recursiveGroupContents.slice(pre.length + m.length);
        return expression.slice(0, startPos) + makeRecursive(pre, post, +gRDepth, true) + expression.slice(startPos + recursiveGroupContents.length);
      } else if (captureName) {
        numCaptures++;
        groupContentsStartPos.set(String(numCaptures), token.lastIndex);
        groupContentsStartPos.set(captureName, token.lastIndex);
        openGroups.push({
          num: numCaptures,
          name: captureName
        });
      } else if (m.startsWith("(")) {
        const isUnnamedCapture = m === "(";
        if (isUnnamedCapture) {
          numCaptures++;
          groupContentsStartPos.set(String(numCaptures), token.lastIndex);
        }
        openGroups.push(isUnnamedCapture ? { num: numCaptures } : {});
      } else if (m === ")") {
        openGroups.pop();
      }
    } else if (m === "]") {
      numCharClassesOpen--;
    }
  }
  return expression;
}
function assertMaxInBounds(max) {
  const errMsg = `Max depth must be integer between 2 and 100; used ${max}`;
  if (!/^[1-9]\d*$/.test(max)) {
    throw new Error(errMsg);
  }
  max = +max;
  if (max < 2 || max > 100) {
    throw new Error(errMsg);
  }
}
function assertNoFollowingRecursion(remainingExpression) {
  if (hasUnescaped(remainingExpression, recursiveToken, Context.DEFAULT)) {
    throw new Error("Recursion can only be used once per regex");
  }
}
function makeRecursive(pre, post, maxDepth, isSubpattern) {
  const namesInRecursed = /* @__PURE__ */ new Set();
  if (isSubpattern) {
    forEachUnescaped(pre + post, namedCapturingDelim2, ({ groups: { captureName } }) => {
      namesInRecursed.add(captureName);
    }, Context.DEFAULT);
  }
  const reps = maxDepth - 1;
  return `${pre}${repeatWithDepth(`(?:${pre}`, reps, isSubpattern ? namesInRecursed : null)}(?:)${repeatWithDepth(`${post})`, reps, isSubpattern ? namesInRecursed : null, "backward")}${post}`;
}
function repeatWithDepth(expression, reps, namesInRecursed, direction = "forward") {
  const startNum = 2;
  const depthNum = (i) => direction === "backward" ? reps - i + startNum - 1 : i + startNum;
  let result = "";
  for (let i = 0; i < reps; i++) {
    const captureNum = depthNum(i);
    result += replaceUnescaped(
      expression,
      String.raw`${namedCapturingDelim2}|\\k<(?<backref>[^>]+)>`,
      ({ 0: m, groups: { captureName, backref } }) => {
        if (backref && namesInRecursed && !namesInRecursed.has(backref)) {
          return m;
        }
        const suffix = `_$${captureNum}`;
        return captureName ? `(?<${captureName}${suffix}>` : `\\k<${backref}${suffix}>`;
      },
      Context.DEFAULT
    );
  }
  return result;
}

// src/index.js
function toDetails(pattern2, options) {
  const opts = getOptions(options);
  const tokenized = tokenize(pattern2, opts.flags);
  const onigurumaAst = parse(tokenized, {
    skipBackrefValidation: opts.tmGrammar,
    verbose: opts.verbose
  });
  const regexAst = transform(onigurumaAst, {
    accuracy: opts.accuracy,
    avoidSubclass: opts.avoidSubclass,
    bestEffortTarget: opts.target
  });
  const generated = generate(regexAst, opts);
  let genPattern = atomic(possessive(recursion(generated.pattern)));
  let subpattern;
  if (regexAst._strategy) {
    genPattern = genPattern.replace(/\(\?:\\p{sc=<<}\|(.*?)\|\\p{sc=>>}\)/s, (_, sub) => {
      subpattern = sub;
      return "";
    });
  }
  const result = {
    pattern: genPattern,
    flags: `${opts.hasIndices ? "d" : ""}${opts.global ? "g" : ""}${generated.flags}${generated.options.disable.v ? "u" : "v"}`
  };
  if (regexAst._strategy) {
    result.strategy = { ...regexAst._strategy };
    if (subpattern) {
      result.strategy.subpattern = subpattern;
    }
  }
  return result;
}
function toOnigurumaAst(pattern2, options) {
  return parse(tokenize(pattern2, options?.flags));
}
function toRegExp(pattern2, options) {
  const result = toDetails(pattern2, options);
  if (result.strategy) {
    return new EmulatedRegExp(result.pattern, result.flags, result.strategy);
  }
  return new RegExp(result.pattern, result.flags);
}
var EmulatedRegExp = class _EmulatedRegExp extends RegExp {
  #strategy;
  constructor(pattern2, flags, strategy) {
    super(pattern2, flags);
    if (strategy) {
      this.#strategy = strategy;
    } else if (pattern2 instanceof _EmulatedRegExp) {
      this.#strategy = pattern2.#strategy;
    }
  }
  /**
  Called internally by all String/RegExp methods that use regexes.
  @override
  @param {string} str
  @returns {RegExpExecArray | null}
  */
  exec(str) {
    const useLastIndex = this.global || this.sticky;
    const pos = this.lastIndex;
    const exec = RegExp.prototype.exec;
    if (this.#strategy.name === "line_or_search_start" && useLastIndex && this.lastIndex) {
      this.lastIndex = 0;
      const match = exec.call(this, str.slice(pos));
      if (match) {
        match.input = str;
        match.index += pos;
        this.lastIndex += pos;
      }
      return match;
    }
    const globalRe = useLastIndex ? this : new RegExp(this, `g${this.flags}`);
    if (this.#strategy.name === "not_search_start") {
      let match = exec.call(this, str);
      if (match?.index === pos) {
        globalRe.lastIndex = match.index + 1;
        match = exec.call(globalRe, str);
      }
      return match;
    }
    if (this.#strategy.name === "after_search_start_or_subpattern") {
      let match = exec.call(this, str);
      if (!match) {
        return match;
      }
      if (match.index === pos) {
        return match;
      }
      const reBehind = new RegExp(`(?:${this.#strategy.subpattern})$`);
      while (match) {
        if (reBehind.exec(str.slice(0, match.index))) {
          return match;
        }
        globalRe.lastIndex = match.index + 1;
        match = exec.call(globalRe, str);
      }
      return match;
    }
    return exec.call(this, str);
  }
};
export {
  EmulatedRegExp,
  toDetails,
  toOnigurumaAst,
  toRegExp
};
//# sourceMappingURL=index.mjs.map
