// SPDX-FileCopyrightText: 2023 XWiki CryptPad Team <contact@cryptpad.org> and contributors
//
// SPDX-License-Identifier: AGPL-3.0-or-later

define([
    'jquery',
    '/common/common-interface.js',
    '/common/hyperscript.js',
    '/common/common-icons.js',
    '/common/common-util.js',

], function ($, UI, h, Icons, Util) {
    //var ChainPad = window.ChainPad;
    var History = {};

    History.create = function (common, config) {
        if (!config.$toolbar) { return void console.error("config.$toolbar is undefined");}
        if (History.loading) { return void console.error("History is already being loaded..."); }
        History.loading = true;
        var $toolbar = config.$toolbar;
        var sframeChan = common.getSframeChannel();
        History.readOnly = common.getMetadataMgr().getPrivateData().readOnly || !common.isLoggedIn();

        if (!config.onlyoffice || !config.setHistory || !config.onCheckpoint || !config.onPatch || !config.makeSnapshot) {
            throw new Error("Missing config element");
        }

        var cpIndex = -1;
        var msgIndex = -1;
        var ooMessages = {};
        var msgs;
        var loading = false;
        var currentTime;
        //Defining position here means it can be passed to the showVersion and share functions
        var position;
        //Defining patch here means it can be passed to the snapshot function
        var patch;
        var currentVersion;
        var forward;
        var revertCheckpoint;

        // Get an array of the checkpoint IDs sorted their patch index
        var hashes = config.onlyoffice.hashes;
        var id;
        var sortedCp = Object.keys(hashes).map(Number);

        var getId = function () {
            var cps = sortedCp.length;
            id = sortedCp[cps -1] || -1;
            return id;
        };

        var endWithCp = sortedCp.length &&
                        config.onlyoffice.lastHash === hashes[sortedCp[sortedCp.length - 1]].hash;

        var fillOO = function (messages) {
            ooMessages = {};
            ooMessages[id] = messages;
        };

        if (endWithCp) { cpIndex = 0; }

        var $version, $share;
        var $hist = $toolbar.find('.cp-toolbar-history');
        $hist.addClass('cp-smallpatch');
        $hist.addClass('cp-history-oo');
        var $bottom = $toolbar.find('.cp-toolbar-bottom');
        var Messages = common.Messages;

        var getVersion = function (position, initial, revert) {
            let version = (id === -1 || id === 0) ? 0 : id;
            if (!Object.keys(ooMessages).length) {
                return '0.0';
            }
            if (typeof(position) === "undefined" || position === -1) {
                position = ooMessages[id]?.length || 0;
            } else if (msgs?.length === position && 
            id !== parseInt(Object.keys(hashes)[Object.keys(hashes).length-1]) &&
            !initial && !revert && $(`[data^="${id+1},"][data*=","]`).length > 1) {
                    version = id+1;
                    position = 0;
                }
            return version + '.' + position;
        };

        var getMessages = function(fromHash, toHash, callback) {
            sframeChan.query('Q_GET_HISTORY_RANGE', {
                channel: config.onlyoffice.channel,
                lastKnownHash: fromHash,
                toHash: toHash,
            }, function (err, data) {
                if (err) { return void console.error(err); }
                if (!Array.isArray(data.messages)) { return void console.error('Not an array!'); }

                var isEmptyPatch = function(msg) {
                    return msg?.changes?.length === 2 &&
                    msg.changes.some(c => c.change.includes('64;AgAAA')) &&
                    msg.changes.some(c => c.change.includes('18;BgAAA') || c.change.includes('23;BgAAAD'));
                };

                var messages;
                if (data.messages[0] && hashes[id]?.index > JSON.parse(data.messages?.[0]?.msg).changesIndex+1 ) {
                    messages = [];
                } else if (config.docType() === 'spreadsheet' && toHash === 'NONE') {
                    messages = (data.messages || []);
                } else {
                    messages = (data.messages || []).slice(1);
                }

                if (messages[0] && isEmptyPatch(JSON.parse(messages[0].msg))) {
                    revertCheckpoint = true;
                    messages.splice(0, 1);
                } else if (messages[1] && isEmptyPatch(JSON.parse(messages[1].msg))) {
                    revertCheckpoint = true;
                    messages.splice(1, 1);
                } else {
                    revertCheckpoint = false;
                }

                if (config.debug) { console.log(data.messages); }
                id = typeof(id) !== "undefined" ? id : getId();

                fillOO(messages);
                loading = false;
                callback(null, messages);
            });
        };

        // We want to load a checkpoint (or initial state)
        var loadMoreOOHistory = function () {
            return new Promise((resolve, reject) => {
                if (!Array.isArray(sortedCp)) { 
                    console.error("Wrong type");
                    return reject();
                }

                // Get the checkpoint ID
                id = typeof(id) !== "undefined" ? id : getId();
                var cp = hashes[id];

                // Get the history between "toHash" and "fromHash". This function is using
                // "getOlderHistory", that's why we start from the more recent hash
                // and we go back in time to an older hash

                // We need to get all the patches between the current cp hash and the next cp hash
                
                var nextId = hashes[id+1] ? hashes[id+1] : undefined;
                // Current cp or initial hash (invalid hash ==> initial hash)
                var fromHash = cp?.hash || 'NONE';
                // Next cp or last hash
                var toHash = nextId ? nextId.hash : config.onlyoffice.lastHash;

                getMessages(toHash, fromHash, function (err) {
                    if (err) {
                        console.error(err);
                        reject(err);
                        return;
                    }
                    resolve();
                });
            });
        };

        loadMoreOOHistory();

        var onClose = function () { config.setHistory(false); };
        var onRevert = function () {
            config.onRevert();
        };

        config.setHistory(true);

        $hist.html('').css('display', 'flex');
        $bottom.hide();

        // UI.spinner($hist).get().show();

        var $fastPrev, $fastNext, $next, $prev;

        var updateButtons = function () {
            $fastPrev.show();
            $next.show();
            $prev.show();
            $fastNext.show();
            $hist.find('.cp-toolbar-history-next, .cp-toolbar-history-previous')
                .prop('disabled', '');

            if ((id === -1 || id === 0) && (ooMessages[id]?.length+1 === Math.abs(msgIndex) || !ooMessages[id]?.length &&  id === 0)){
                $prev.prop('disabled', 'disabled');
                $fastPrev.prop('disabled', 'disabled');
            }
            var version = currentVersion.split('.');
            var hashesLength = Object.keys(hashes).length;

            if (currentVersion === Messages.oo_version_latest || hashesLength === parseInt(version[0]) && ooMessages[id]?.length === parseInt(version[1]) ||
            hashesLength+1 === id && (msgIndex === -1) && forward  ||
            hashes[hashesLength-1] === id && !ooMessages[id].length && msgIndex === 0) {
                $next.prop('disabled', 'disabled');
                $fastNext.prop('disabled', 'disabled');
            }
        };

        var loadingFalse = function () {
            setTimeout(function () {
                $('iframe').blur();
                loading = false;
            }, 200);
        };

        var showVersion = function (initial, revert, empty) {
            $('.cp-history-timeline-pos-oo').remove();
            $('.cp-history-oo-timeline-pos').removeClass('cp-history-oo-timeline-pos');

            var currentPatch;
            if (initial) {
                currentPatch = $('.cp-history-patch').last();
                currentVersion = getVersion(position, initial);
            } else if (empty) {
                currentPatch = $(`[data="${id},0"]`);
                currentVersion = getVersion(position, initial);
            } else if ($(`[data="${id},${position}"]`).length) {
                currentPatch = $(`[data="${id},${position}"]`);
                currentVersion = getVersion(position, initial, true);
            } else if (msgs?.length === position && 
            id !== parseInt(Object.keys(hashes)[Object.keys(hashes).length-1]) ) {
                currentPatch = $(`[data="${id+1},0"]`);
                currentVersion = getVersion(position, initial);
            } 

            if (initial || position === msgs?.length && (id === -1 || 
            id === Object.keys(hashes)[Object.keys(hashes).length-1])) { 
                currentVersion = Messages.oo_version_latest; 
            }

            var patchTime = patch ? new Date(patch.time).toLocaleString() : '';
            $version.text(Messages.oo_version + currentVersion + ' ' + patchTime);
            var pos = Icons.get('chevron-down', {'class': 'cp-history-timeline-pos-oo'});
            $(currentPatch).addClass('cp-history-oo-timeline-pos').append(pos);

            updateButtons();
            loadingFalse();
        };

        var displayCheckpointTimeline = function(initial) {          
            var bar = $hist.find('.cp-history-timeline-container');
            $(bar).addClass('cp-history-timeline-bar').addClass('cp-oohistory-bar-el');
            
            msgs = ooMessages[id];
            if (initial) {
                var snapshotsEl = [];
                var msgsRev = msgs;
            } else {
                snapshotsEl = Array.from($hist.find('.cp-history-snapshots')[0].childNodes);
                msgsRev = msgs.slice().reverse();
            }

            var cpNfInner = common.startRealtime(config);
            var md = Util.clone(cpNfInner.metadataMgr.getMetadata());
            var snapshots = md.snapshots;

            var patchWidth;
            var patchDiv;
            var firstCp = (id === 0) ? true : false;
            for (var i = 0; i < msgsRev.length; i++) {
                var msg = msgs[i];
                if (initial || id === -1) {
                    patchWidth = (1/msgs?.length)*100;
                } else {
                    patchWidth = (1/(msgs?.length+Array.from($hist.find('.cp-history-snapshots')[0].childNodes).length))*100;
                }
                
                patchDiv = h('div.cp-history-patch', {
                    style: 'width:'+patchWidth+'%;',
                    title: new Date(msgsRev[i].time).toLocaleString(),
                    data: [id, msgsRev.indexOf(msg)] 
                });
                if (initial) {
                    snapshotsEl.push(patchDiv);
                } else {
                    snapshotsEl.unshift(patchDiv);
                }
                 if (snapshots) {
                    var match = Object.values(snapshots).find(item => item.time === msg.time);
                    if (match) { $(patchDiv).addClass('cp-history-snapshot').append(Icons.get('snapshot', {title: match.title})); }
                 }
            }

            var finalpatchDiv = h('div.cp-history-patch', {
                style: 'width:'+patchWidth+'%; height: 100%; position: relative',
                title: new Date().toLocaleString(),
                data: [id, msgs?.length] 
            });
            if (initial) {
                snapshotsEl.push(finalpatchDiv);
            } else  {
                firstCp ? snapshotsEl.splice(msgs?.length, 0, finalpatchDiv) : snapshotsEl.splice(msgs?.length-1, 0, finalpatchDiv);
            }

            if (!msgsRev.length && !Object.keys(hashes).length || initial && !msgs?.length) {
                $(finalpatchDiv).css('width', '100%');
            } else {
                $(finalpatchDiv).css('width', `${($(snapshotsEl[snapshotsEl.indexOf(finalpatchDiv)+1])?.width()/ $(snapshotsEl[snapshotsEl.indexOf(finalpatchDiv)+1])?.parent().width())*100}%`);
                patchWidth = ($(snapshotsEl[snapshotsEl.indexOf(finalpatchDiv)+1])?.width()/ $(snapshotsEl[snapshotsEl.indexOf(finalpatchDiv)+1])?.parent().width())*100;
            }
                
            var pos = Icons.get('chevron-down', {'class': 'cp-history-timeline-pos-oo'});

            var patches = h('div.cp-history-snapshots.cp-history-snapshots-oo', [
                snapshotsEl
            ]);
            $(patches).css('height', '100%');
            $(patches).css('display', 'flex');

            bar.html('').append([
                patches
            ]);

            if (snapshotsEl.length === 1) {
                $('.cp-history-patch').css('width', '100%');
            }
            if (!initial) {
                var finalPatchWidth = patchWidth ? patchWidth : 100/$hist.find('.cp-history-snapshots')[0].childNodes.length;
                Array.from($hist.find('.cp-history-snapshots')[0].childNodes).forEach(function(patch) {
                    $(patch).css('width', `${finalPatchWidth}%`);
                });
            } 
            if (initial) {
                $('.cp-history-patch').last().addClass('cp-history-oo-timeline-pos').append(pos);
            }

            $('.cp-history-patch').on('click', function(e) {
                var patchData = $(e.target).attr('data').split(',');

                var cpNo = parseInt(patchData[0]);
                var patchNo = parseInt(patchData[1]);
                id = cpNo;

                loadMoreOOHistory().then(() => {
                    msgs = ooMessages[id];
                    if (cpNo === -1) {
                        var q = msgs.slice(0, patchNo);
                        config.onPatchBack({}, q);
                        patch = msgs[patchNo];
                        position = (patchNo === msgs?.length) ? msgs?.length : msgs.indexOf(patch);
                        msgIndex = position === -1 ? -1 : position - msgs?.length-1;
                        showVersion(false, true);
                        updateButtons();
                        return;
                    } else if (cpNo === 0 && patchNo === 0) {
                        config.onPatchBack({});
                        patch = msgs[0];
                    } else if (!msgs?.length ) {
                        q = msgs.slice(0, patchNo);
                        config.onPatchBack(hashes[cpNo], q);
                        patch = msgs[msgs?.length-1];
                        position = patch ? msgs.indexOf(patch)+1 : 0;
                        msgIndex = position === -1 ? -1 : position - msgs?.length-1;
                        showVersion(false, false, true);
                        updateButtons();
                        return;
                    } else if (patchNo === msgs?.length && msgs?.length < $(`[data^="${id},"][data*=","]`).length) {
                        q = msgs.slice(0, patchNo);
                        config.onPatchBack(hashes[cpNo], q);
                        patch = msgs[msgs?.length-1];
                        position = patch ? msgs.indexOf(patch)+1 : 0;
                        msgIndex = position === -1 ? -1 : position - msgs?.length-1;
                        showVersion(false, true);
                        updateButtons();
                        return;
                    } else  {
                        q = msgs.slice(0, patchNo);
                        config.onPatchBack(hashes[cpNo], q);
                        patch = msgs[patchNo];
                    }
                    position = patch ? msgs.indexOf(patch) : 0;
                    msgIndex = position === -1 ? -1 : position - msgs?.length-1;
                    showVersion(false);
                    updateButtons();
                });
            });
        };

        var restore;

        var next = async function () {
            forward = true;
            msgIndex++;
            msgs = ooMessages[id];
            var hasHashes = Object.keys(hashes).length;

            if (hasHashes) {
                // /Check if the end of the checkpoint has been reached and the next one should be loaded
                if (msgIndex === 0) {
                    id++;
                    await loadMoreOOHistory();
                    msgs = ooMessages[id];

                    // Empty checkpoint (checkpoint created/history restored with no further changes)
                    if (!msgs?.length) {
                        config.loadHistoryCp(hashes[id]);
                        msgs = ooMessages[id];
                        msgIndex = -msgs?.length;
                        patch = msgs[0];
                        position = 0;
                        showVersion(false);
                        return;
                    }
                    //Is the checkpoint the result of restoring history? If yes, we need to load an extra patch
                    if (revertCheckpoint ) { 
                        msgIndex = -msgs?.length;
                        config.onPatchBack(hashes[id], [msgs[0]]);
                        position = 1;
                        showVersion(false);
                    } else {
                        msgIndex = -msgs?.length - 1;
                        config.loadHistoryCp(hashes[id]);
                        position = 0;
                        showVersion(false, true);
                    }
                    return;
                }

                if (!msgs?.length) {
                    position = 0;
                    msgIndex = -1;
                    id++;
                    showVersion(false);
                    return config.loadHistoryCp(hashes[id]);
                }
                // Adjust msgIndex after fastPrev
                if (Math.abs(msgIndex) > msgs?.length) { msgIndex = -msgs?.length; }
            }
            else if (msgs?.length + msgIndex === -1) { msgIndex++; }

            patch = msgs[msgs?.length + msgIndex];
            position = msgs.indexOf(patch) + 1;
            config.onPatch?.(patch);
            msgIndex === -1
                ? showVersion(false, true)
                : showVersion(false);
        };

        var prev = function () {
            forward = false;
            msgs = ooMessages[id];
            let hasHashes = Object.keys(hashes).length;
            let cp = hasHashes ? hashes[id] : {};
            let loadPrevCp = (!msgs?.length) ||
                    (msgs?.length + 1 === Math.abs(msgIndex) && id !== 0) ||
                    (msgs?.length - Math.abs(msgIndex) === -2); 
            var isRevert = revertCheckpoint;

            //Check if the end of the checkpoint has been reached and the previous one should be loaded
            if (hasHashes && loadPrevCp) {
                id--; 
                msgIndex = -1;
                return loadMoreOOHistory().then(() => {
                    msgs = ooMessages[id];
                    //Empty checkpoint - checkpoint saved with no further changes
                    if (!msgs?.length) {
                        msgs = ooMessages[id];
                        config.onPatchBack(hashes[id], msgs.slice(0, msgIndex));
                        msgIndex--;
                        patch = msgs[msgs?.length-1];
                        position = msgs.indexOf(patch);
                        if (!$(`[data="${id},0"]`).length) {
                            displayCheckpointTimeline();
                        }
                        showVersion(false, false, true);
                        return;
                    }
                    cp = hashes[id];
                    var q = msgs.slice(0, msgIndex);
                    patch = msgs[msgs?.length-1];

                    //Is the checkpoint the result of restoring history? If yes, we need to load an extra patch
                    if (isRevert) {
                        config.onPatchBack(cp, q); 
                        msgIndex--;
                    } else {
                        restore = true;
                        if (!$(`[data="${id},${position}"]`).length) {
                            displayCheckpointTimeline(false, true);
                        }
                        config.onPatchBack(cp, msgs);
                        patch = msgs[msgs?.length-1];
                        position = msgs?.length;
                    }

                    //Check if this checkpoint has already been added to the timeline
                    if (!$(`[data="${id},0"]`).length) {
                        displayCheckpointTimeline();
                    }
                    showVersion(false, restore);
                    restore = false;
                });
            }
            var q = msgs.slice(0, msgIndex);
            config.onPatchBack(cp, q);   
            patch = msgs[msgs?.length + msgIndex];
            msgIndex--; 
            position = msgs.indexOf(patch);            
            showVersion(false);
            
        };

        setTimeout(() => {
            displayCheckpointTimeline(true);
            showVersion(true);
        }, "1000");

        // Create the history toolbar
        var display = function () {
            $hist.html('');

            var fastPrev = h('button.cp-toolbar-history-previous', { title: Messages.history_prev }, [
                Icons.get('history-fast-prev'),
            ]);
            var fastNext = h('button.cp-toolbar-history-next', { title: Messages.history_next }, [
                Icons.get('history-fast-next'),
            ]);
            var _next = h('button.cp-toolbar-history-next', { title: Messages.history_next }, [
                Icons.get('history-next'),
            ]);
            var _prev = h('button.cp-toolbar-history-previous', { title: Messages.history_prev }, [
                Icons.get('history-prev')
            ]);
            $fastPrev = $(fastPrev);
            $prev = $(_prev);
            $fastNext = $(fastNext).prop('disabled', 'disabled');
            $next = $(_next).prop('disabled', 'disabled');

            var time = h('div.cp-history-timeline-time');
            var version = h('div.cp-history-timeline-version');
            $version = $(version);
            var timeline = h('div.cp-toolbar-history-timeline', [
                h('div.cp-history-timeline-line', [
                    h('span.cp-history-timeline-container')
                ]),
                h('div.cp-history-timeline-actions', [
                    h('span.cp-history-timeline-prev', [
                        fastPrev,
                        _prev
                    ]),
                    time,
                    version,
                    h('span.cp-history-timeline-next', [
                        _next,
                        fastNext
                    ])
                ])
            ]);
            var snapshot = h('button', {
                title: Messages.snapshots_new,
                class: 'cp-history-create-snapshot'
            }, [
                Icons.get('snapshot')
            ]);
            var share = h('button', { title: Messages.history_shareTitle }, [
                Icons.get('share'),
                h('span', Messages.shareButton)
            ]);
            var restore = h('button', {
                title: Messages.history_restoreTitle,
            }, [
                Icons.get('history-restore'),
                h('span', Messages.history_restore)
            ]);
            var close = h('button', { title: Messages.history_closeTitle }, [
                Icons.get('close'),
                h('span', Messages.history_close)
            ]);
            var actions = h('div.cp-toolbar-history-actions', [
                h('span.cp-history-actions-first', [
                    snapshot,
                    share
                ]),
                h('span.cp-history-actions-last', [
                    restore,
                    close
                ])
            ]);

            if (History.readOnly) {
                snapshot.disabled = true;
                restore.disabled = true;
            }

            $share = $(share);
            $hist.append([timeline, actions]);

            var onKeyDown, onKeyUp;
            var closeUI = function () {
                $hist.hide();
                $bottom.show();
                $(window).trigger('resize');
                $(window).off('keydown', onKeyDown);
                $(window).off('keyup', onKeyUp);
            };

            // Push one patch
            $next.click(function () {
                if (loading) { return; }
                loading = true;
                next();
            });
            $prev.click(function () {
                if (loading) { return; }
                loading = true;
                prev();
            });

            // Go to next checkpoint
            $fastNext.click(function () {
                if (loading) { return; }
                loading = true;
                var msgs = ooMessages[id];
                if (id < Object.keys(hashes).length && id !== -1) {
                    if (id === -1) {
                        id = 1;
                    } else {
                        id++;
                    }
                    loadMoreOOHistory().then(() => {
                        var cp = hashes[id];
                        config.loadHistoryCp(cp);
                        var msgs = ooMessages[id];
                        msgIndex = -msgs?.length-1;
                        position = 0;
                        showVersion(false);
                        loadingFalse();
                        return;
                    });
                } 
                else {
                    var cp = hashes[id];
                    msgs = ooMessages[id];
                    msgIndex = -1;
                    config.onPatchBack(cp, msgs);
                }
                loadingFalse();
                position = msgs?.length;
                showVersion(false);
            });
            
            // Go to previous checkpoint
            $fastPrev.click(function () {
                if (loading) { return; }
                loading = true;
                if (!ooMessages[id].length || ooMessages[id].length+1 === Math.abs(msgIndex)) {
                    id--;
                } 
                var cp = hashes[id];
                config.loadHistoryCp(cp);
                loadMoreOOHistory().then(() => {
                    var msgs = ooMessages[id];
                    msgIndex = -msgs?.length-1;
                    if (!$(`[data="${id},0"]`).length) {
                        displayCheckpointTimeline();
                    }
                    patch = msgs[msgs?.length-1];
                    position = 0;

                    showVersion(false);
                    updateButtons(true);
                });
                
                loadingFalse();
            });
            onKeyDown = function (e) {
                var p = function () { e.preventDefault(); };
                if ([38, 39].indexOf(e.which) >= 0) { p(); return $next.click(); } // Right
                if (e.which === 33) { p(); return $fastNext.click(); } // PageUp
                if (e.which === 34) { p(); return $fastPrev.click(); } // PageUp
                if (e.which === 27) { p(); return $(close).click(); }
            };
            onKeyUp = function (e) { e.stopPropagation(); };
            $(window).on('keydown', onKeyDown).on('keyup', onKeyUp).focus();

            // Versioned link
            $share.click(function () {
                common.getSframeChannel().event('EV_SHARE_OPEN', {
                    versionHash: getVersion(position)
                });
            });
            $(snapshot).click(function () {
                if (cpIndex === -1 && msgIndex === -1) { return void UI.warn(Messages.snapshots_ooPickVersion); }
                var input = h('input', {
                    placeholder: Messages.snapshots_placeholder
                });
                var $input = $(input);
                var content = h('div', [
                    h('h5', Messages.snapshots_new),
                    input
                ]);

                var buttons = [{
                    className: 'cancel',
                    name: Messages.filePicker_close,
                    onClick: function () {},
                    keys: [27],
                }, {
                    className: 'primary',
                    iconClass: 'snapshot',
                    name: Messages.snapshots_new,
                    onClick: function () {
                        var val = $input.val();
                        if (!val) { return true; }
                        msgs = ooMessages[id];
                        config.makeSnapshot(val, function (err) {
                            if (err) { return; }
                            $input.val('');
                            UI.log(Messages.saved);
                        }, {
                            hash: getVersion(position),                            
                            time: currentTime || patch && patch.time || 0
                        });
                    },
                    keys: [13],
                }];

                UI.openCustomModal(UI.dialog.customModal(content, {buttons: buttons }));
                setTimeout(function () {
                    $input.focus();
                });
            });

            // Close & restore buttons
            $(close).click(function () {
                History.loading = false;
                onClose();
                closeUI();
            });
            $(restore).click(function () {
                UI.confirm(Messages.history_restorePrompt, function (yes) {
                    if (!yes) { return; }
                    closeUI();
                    History.loading = false;
                    onRevert();
                    UI.log(Messages.history_restoreDone);
                });
            });
        };

        display();

    };

    return History;
});


